!----------------------------------------------------------------
! <CONTACT EMAIL="thomas.neumann@io-warnemuende.de"> T. Neumann 
! </CONTACT>
! <CONTACT EMAIL="hagen.radtke@io-warnemuende.de"> H. Radtke 
! </CONTACT>
! 
! <REVIEWER EMAIL="none yet"> 
! </REVIEWER>
!<OVERVIEW>
! Ecological Regional Ocean Model - ERGOM 
!</OVERVIEW>
!
!<DESCRIPTION>
!       ERGOM is developed for marine ecosystems in which bentho-pelagic
!       fluxes and changing redox conditions are important.It describes
!       the nitrogen and phosphorus cycle and partly the sulfor cycle.
!       Included is a carbon module and the oxygen dynamics.
!</DESCRIPTION>
!
! <INFO>
! http://www.ergom.net
! <REFERENCE>
! T. Neumann, 2000: Towards a 3D-ecosystem model of the Baltic Sea
! Journal of Marine Systems, 25, 405-419
! </REFERENCE>
! <REFERENCE>
! T. Neumann, W. Fennel, Ch. Kremp 2002: Experimental Simulations with
! an Ecosystem Model of the Baltic Sea: A Nutrient Load Reduction Experiment,
! Global Biogeochemical Cycles 16, No 3, 7-1  7-19
! </REFERENCE>
! <REFERENCE>
! W. Fennel, Th. Neumann, 2004: Introduction to the Modelling of Marine Ecosystems,
! Elsevier Oceanography Series 72, Elsevier
! </REFERENCE>
! <REFERENCE>
! Radtke, H., T. Neumann, M. Voss, and W. Fennel (2012), Modeling pathways of
! riverine nitrogen and phosphorus in the Baltic Sea,
! J. Geophys. Res., 117, C09024, doi:10.1029/2012JC008119
! </REFERENCE>
! <REFERENCE>
! I. Kuznetsov, T. Neumann, Simulation of carbon dynamics in the Baltic Sea with a 3D model,
! Journal of Marine Systems, Available online 22 October 2012,
! ISSN 0924-7963, doi: 10.1016/j.jmarsys.2012.10.011.
! </REFERENCE>
! <DEVELOPER_NOTES>
! Ergom will be continuously developed.
! </DEVELOPER_NOTES>
! </INFO>
!
!----------------------------------------------------------------

module generic_ERGOM

  use coupler_types_mod,   only: coupler_2d_bc_type
  use field_manager_mod,   only: fm_string_len
  use fms_mod,             only: open_namelist_file, close_file, check_nml_error  
  use mpp_mod,             only: mpp_error, NOTE, WARNING, FATAL, stdout, stdlog
  use mpp_mod,             only: mpp_clock_id, mpp_clock_begin, mpp_clock_end, CLOCK_ROUTINE
  use time_manager_mod,    only: time_type, get_date, days_in_year
  use fm_util_mod,         only: fm_util_start_namelist, fm_util_end_namelist  
  use diag_manager_mod,    only: register_diag_field, send_data 

  use g_tracer_utils, only : g_tracer_type,g_tracer_start_param_list,g_tracer_end_param_list
  use g_tracer_utils, only : g_tracer_add,g_tracer_add_param, g_tracer_set_files
  use g_tracer_utils, only : g_tracer_set_values,g_tracer_get_pointer,g_tracer_get_common
  use g_tracer_utils, only : g_tracer_coupler_set,g_tracer_coupler_get
  use g_tracer_utils, only : g_tracer_send_diag, g_tracer_get_values  
  use g_tracer_utils, only : g_diag_type, g_diag_field_add


  implicit none ; private

  character(len=fm_string_len), parameter :: mod_name       = 'generic_ERGOM'
  character(len=fm_string_len), parameter :: package_name   = 'generic_ergom'

  public do_generic_ERGOM
  public generic_ERGOM_register
  public generic_ERGOM_init
  public generic_ERGOM_update_from_coupler
  public generic_ERGOM_update_from_source
  public generic_ERGOM_update_from_bottom
  public generic_ERGOM_set_boundary_values
  public generic_ERGOM_register_diag
  public generic_ERGOM_end

  !
  !This type contains all the parameters and arrays used in this module.
  !
  !Note that there is no programatic reason for treating
  !the following as a type. These are the parameters used only in this module. 
  !It suffices for varables to be a declared at the top of the module. 
  !nnz: Find out about the timing overhead for using type%x rather than x

  !An auxiliary type for storing varible names
  type, public :: vardesc
     character(len=fm_string_len) :: name     ! variable name in a NetCDF file.
     character(len=fm_string_len) :: longname ! long name of that variable.
     character(len=1)  :: hor_grid            ! hor. grid:  u, v, h, q, or 1.
     character(len=1)  :: z_grid              ! vert. grid:  L, i, or 1.
     character(len=1)  :: t_grid              ! time description: s, a, m, or 1.
     character(len=fm_string_len) :: units    ! dimensions of the variable.
     character(len=1)  :: mem_size            ! size in memory: d or f.
  end type vardesc
  
  type generic_ERGOM_type
     ! Parameters for gas exchange follow
     real :: t1_nit, t2_nit, t3_nit, t4_nit
     real :: e1_nit
     real :: s1_nit, s2_nit, s3_nit          ! [1]
     real :: a1_nit, a2_nit, a3_nit, a4_nit  ! coefficients for Schmidt number [1/psu]

     real :: t0_o2, t1_o2, t2_o2, t3_o2, t4_o2, t5_o2  ! for oxygen solubility
     real :: b0_o2, b1_o2, b2_o2, b3_o2, c0_o2         ! for oxygen solubility
     real :: a1_o2 , a2_o2 , a3_o2 , a4_o2   !  coefficients for Schmidt number [1/psu]

     real :: a1_co2, a2_co2, a3_co2, a4_co2  !  coefficients for Schmidt number [1/psu]
     ! End of parameters for gas exchange

<tracers solubility/=0>
     real, ALLOCATABLE, dimension(:,:) :: &
          <gasName>_csurf, &                       ! surface concentration of <gasName> [mol/m3]
          <gasName>_alpha, &                       ! surface solubility of <gasName>    [mol/m3/Pa]
          <gasName>_sc_no                          ! Schmidt number of <gasName>        [1]
</tracers>

     real, ALLOCATABLE, dimension(:,:) :: &
<processes vertLoc=SED>
          <name> , &                      ! <description>
</processes>
<processes vertLoc=SUR>
          <name> , &                      ! <description>
</processes>
<auxiliaries vertLoc=SED>
          <name> , &                      ! <description>
</auxiliaries>
<auxiliaries vertLoc=SUR>
          <name> , &                      ! <description>
</auxiliaries>
<auxiliaries isZIntegral=1>
          zintegralarray_<name> , &       ! place to accumulate <description> over k layers
</auxiliaries>
<tracers vertLoc=FIS>
          cumulated_change_of_<name> , &  ! place to accumulate changes in <description> over k layers
</tracers>
          host_biores                     ! bioresuspension rate [mol/m2/s]
     real, ALLOCATABLE, dimension(:,:,:) :: &
<auxiliaries vertLoc=WAT>
          <name> , &                      ! <description>
</auxiliaries>
<processes vertLoc=WAT>
          <name> , &                      ! <description>
</processes>
<celements isAging/=0>
          old_<aged>  , &  ! temporary storage of age concentration of <description>
</celements>
          irr_inst , &                    ! instantanuous radiation [W/m2]
          irr_zw , &                      ! instantanuous radiation at bottom of cell [W/m2]
          bio_opacity                     ! opacity of biological tracers [1/m]
<tracers vertLoc=WAT; vertSpeed=0>
     real, dimension(:,:,:,:), pointer :: &
          <name>          ! <description>
</tracers>
<tracers vertLoc=WAT; vertSpeed/=0>
     real, dimension(:,:,:), pointer :: &
          vmove_<name> , & ! vertical movement [m/s]
          vdiff_<name>     ! vertical diffusion [m2/s]
</tracers>

     real    :: gamma_1                  ! constant for opacity contribution of chlorophyll [kg/g/m]
     real    :: rho_0                    ! water density for sediment module [kg/m3]
     integer :: & 
<processes>
          id_<name> , &                  ! diag id for <description>
</processes>
<auxiliaries>
          id_<name> , &                  ! diag id for <description>
</auxiliaries>
<tracers vertLoc=WAT; atmosDep/=0>
          id_dep_wet_<name>     = -1,  & ! <description> atmospheric deposition
</tracers>
<tracers vertLoc=WAT; riverDep/=0>
          id_runoff_flux_<name> = -1,  & ! <description> runoff flux to the ocean
          id_diffusive_flux_<name> = -1,  & ! <description> diffusive flux from land to the ocean
</tracers>
          id_irr_inst   = -1             ! instantaneous light [W/m2]
     character(len=fm_string_len) :: ice_restart_file
     character(len=fm_string_len) :: ocean_restart_file, IC_file
  end type generic_ERGOM_type 

  type spm_type
     ! This is a type which describes a type of suspended particulate matter
     ! that is able to settle (such as detritus).
     ! You should specify "sediment_to" to allow sedimentation to a sed_type tracer.
     real, ALLOCATABLE, dimension(:,:,:)  :: &
	  move  , &                        ! vertical velocity (<0 for sinking) [m/s]
       diff                             ! diffusivity [m2/s]
     real, dimension(:,:,:,:), pointer :: &
          p_wat                         ! pointer to 3d variable for concentration in water column [mol/kg]
     real, ALLOCATABLE, dimension(:,:) :: &
          jsed                          ! sedimentation [mol/m2/s]
     real, ALLOCATABLE, dimension(:,:) :: &
          btf                           ! The total bottom flux [mol/m2/s]
     real :: &
	  wsink0          = 0. , &      ! sinking velocity (<0 for sinking) [m/d]
          wsed            = 0.          ! sedimentation rate [m/d]
     character(len=32) ::  &
          name          = 'none' , &    ! name of 3d tracer
          longname      = 'none' , &    ! long name for output
          sediment_to   = 'none'        ! to which sed_type 2d variable the sedimentation takes place, 
     integer ::               &
          index_sediment_to , &         ! Index i in the array sed(i) to which the sedimentation takes place
          id_jsed                       ! Diag id for sedimentation [mol/m2/s]
  end type spm_type
  
  type sed_type
     ! This is a type which describes a type of particles in the sediment 
     ! that is able to be resuspended (such as detritus).
     ! You should specify "suspend_to" to allow resuspension to a spm_type variable. 
     real, ALLOCATABLE, dimension(:,:,:) :: &
          f_sed                         ! f_sed(i, j, layer) -> 2d arrays for storing sediment concentration [mol/m2]
     real, ALLOCATABLE, dimension(:,:) :: &
          jgain_sed , &                 ! gain by transformation of other sediments [mol/m2/s]
          jloss_sed , &                 ! loss by transformation of sediment [mol/m2/s]
          jres      , &                 ! resuspension [mol/m2/s]
          jbiores                       ! bioerosion [mol/m2/s]
     real :: &
          erosion_rate    = 0. , &      ! erosion rate [1/d]
          bioerosion_rate = 0. , &      ! erosion rate by benthic animals [1/d]
          molar_volume    = 0. , &      ! volume of this tracer when deposited in the sediment [m3/mol]
          critical_stress = 160.        ! critical shear stress when erosion starts [N/m2]
     character(len=32) ::  &
          name          = 'none' , &    ! name of 2d tracer
          longname      = 'none' , &    ! long name for output
          suspend_to    = 'none'        ! to which spm_type 3d variable the resuspension takes place
     integer ::               &
          index_suspend_to  , &         ! Index i in the array spm(i) to which the resuspension takes place
          id_jgain_sed , &              ! Diag id for gain by transformation of other sediments [mol/m2/s]
          id_jloss_sed , &              ! Diag id for loss by transformation of sediment [mol/m2/s]
          id_jbiores   , &              ! Diag id for bio-resuspension [mol/m2/s]
          id_jres                       ! Diag id for resuspension [mol/m2/s]
  end type sed_type

  type sed_defs_type
    integer :: &
          NUM_LAYERS        = -1 , & ! Number of vertical layers
          layer_propagation = -1 , & ! Sediment layer propagation settings, 
	                             ! SLP_DOWNWARD=1, SLP_FULL_BOX=2, SLP_OLD_ERGOM=3
          erosion_mode      = -1     ! Sediment erosion mode, INDEPENDENT=1, MAXSTRESS=2, ORGANIC=3
    real, allocatable, dimension(:) :: layer_height  ! (maximum) height of vertical layers [m]. 
                                                     ! <0 means the layer may become infinitely thick.
  end type sed_defs_type

  type tracer_2d
    character(len=fm_string_len) ::  name               = 'none'  ! name of the 2d tracer
    character(len=fm_string_len) ::  longname           = 'none'  ! longname of the 2d tracer
    character(len=fm_string_len) ::  units              = 'none'  ! units of the 2d tracer
    character(len=fm_string_len) ::  name_of_3d_tracer  = 'none'  ! name of the 3d tracer
    integer                      ::  layer_in_3d_tracer = 0       ! z-level inside the 3d tracer
    integer                      ::  diag_field         = -1      ! handler returned from register_diag_field
    real, dimension(:,:), pointer::  p_field                      ! pointer to the 2d field stored
    logical                      ::  field_assigned     = .false. ! whether a 2d field has been assigned
  end type tracer_2d
  
  ! Sediment layer propagation settings
  integer, parameter             :: SLP_DOWNWARD=1, SLP_FULL_BOX=2, SLP_OLD_ERGOM=3
  ! Sediment erosion settings
  integer, parameter             :: INDEPENDENT=1, MAXSTRESS=2, ORGANIC=3

<constants>
  real :: <name> = <value> ! <description>
</constants>

  type(generic_ERGOM_type), save   :: ergom

  type(sed_defs_type),                                   save :: sed_defs
  type(spm_type),         ALLOCATABLE, dimension(:), save :: array_spm
  type(sed_type),         ALLOCATABLE, dimension(:), save :: array_sed

  type(tracer_2d),            ALLOCATABLE, dimension(:), save :: tracers_2d
   

! Most ecosystem parameters use 1/d as time unit. This is to convert  
  real, parameter :: sperd = 24.0 * 3600.0
  real, parameter :: watt_2_my_einstein = 4.6
  real, parameter :: missing_value1=-1.0e+20
    
  !The following logical for using this module is overwritten 
  ! by generic_tracer_nml namelist
  logical, save :: do_generic_ERGOM = .false.
  
  integer, parameter :: maxspm=100, maxsed=100, max_sediment_layers=2
  integer, parameter :: maxspmm2=maxspm-2, maxsedm2=maxsed-2
  real, dimension(maxspm) :: &
        wsink0_spm         , &   ! sinking velocity (<0 for sinking) [m/d]
        wsed_spm                 ! sedimentation rate [m/d]
  real, dimension(maxsed) :: &
        erosion_rate_sed   , &   ! erosion rate [1/d]
        bioerosion_rate_sed, &   ! erosion rate by benthic animals [1/d]
        molar_volume_sed   , &   ! volume of this tracer when deposited in the sediment [m3/mol]
        critical_stress_sed      ! critical shear stress when erosion starts [N/m2]
  
  character(len=32) ::           &
        name_spm(maxspm)    , & ! name of this type of spm (suspended particulate matter)
        sediment_to(maxspm) , & ! name of the sed(:) tracer to which sedimentation takes place
        longname_spm(maxspm)	! long name for output

  character(len=32) ::           &
        name_sed(maxsed)     , & ! name of this type of sedimented matter
        suspend_to(maxsed)   , & ! name of spm(:) tracer to which the resuspension takes place
        longname_sed(maxsed)	 ! long name for output
                                  
  real, dimension(max_sediment_layers) :: &
        sed_layer_height         ! (maximum) height of vertical layers [m]. 
	                           ! < 0: the layer may become infinitely thick.
  real :: gamma_1 = 0.0150943    ! constant for opacity contribution of chlorophyll [kg/g/m]

  integer :: NUM_SPM     = 1  
  integer :: NUM_SED     = 1  
  integer :: NUM_VMOVE_STEPS       = 1
  integer :: NUM_SEDIMENT_LAYERS   = 2
  integer :: sed_layer_propagation = 1 ! Sediment layer propagation definitions, 
                                            ! SLP_DOWNWARD=1, SLP_FULL_BOX=2, SLP_OLD_ERGOM=3
  integer :: sed_erosion_mode      = 1 ! Sediment erosion mode, INDEPENDENT=1, MAXSTRESS=2, ORGANIC=3
          
  integer :: n, m
  integer :: vlev_sed = 2                ! number of 2d tracers that may be stored in one diagnostic 3d tracer            

  logical :: genus_style_par = .false.
  
  ! identification numbers for mpp clocks
  integer :: id_source, id_init, id_susp, id_alloc, id_preloop, id_mainloop, id_vertmig, id_output

  data(wsink0_spm (n) , n=1, maxspm) /-3.0, -1.0, maxspmm2*0.0/	  ! sinking velocity (<0 for sinking) [m/d]
  data(wsed_spm   (n) , n=1, maxspm) / 2.5,  0.5, maxspmm2*0.0/	  ! sedimentation rate [m/d]
  data(name_spm(n)    , n=1, maxspm) /'det','ipw', maxspmm2*'none'/	  ! name of spm tracer
  data(sediment_to(n) , n=1, maxspm) /'none','none', maxspmm2*'none'/	  ! name of sed tracer to which sedimentation takes place
  
  data(name_sed(n)    , n=1, maxsed) /'sed','ips', maxsedm2*'none'/	  ! name of sed tracer
  data(suspend_to(n)  , n=1, maxsed) /'none','none', maxsedm2*'none'/	  ! name of spm tracer to which resuspension takes place
  data(longname_sed(n), n=1, maxsed) /'detritus','iron phosphate', maxsedm2*'none'/ 
                                                          ! long name for output
  data(erosion_rate_sed(n), n=1, maxsed) /6.0, 6.0, maxsedm2*0.0/       ! erosion rate [1/d]
  data(bioerosion_rate_sed(n), n=1, maxsed) /0.0, 0.0, maxsedm2*0.0/    ! erosion rate by benthic animals [1/d]
  data(molar_volume_sed(n),n=1, maxsed) /0.01111, 0.0, maxsedm2*0.0/    ! molar tracer volume [m3/mol]
  data(critical_stress_sed(n),n=1, maxsed) /0.196, 0.196, maxsedm2*0.0/ ! critical shear stress when erosion starts [N/m2]
  
  data(sed_layer_height(n),n=1,max_sediment_layers) /0.02222, -1.0/  
                                                          ! (maximum) height of vertical layers [m]. 
                                                          ! <0 mean the layer may become infinitely thick.
 
  namelist /ergom_nml/  &
! constants for biomodel
<constants>
   <name>       , & ! <description>
</constants>
   NUM_SPM      , &
   NUM_SED      , &
   NUM_SEDIMENT_LAYERS , &
   NUM_VMOVE_STEPS     , & 
! suspended particulate matter parameters
   name_spm       , & ! name of spm tracer
   longname_spm   , & ! long name for output
   wsink0_spm     , & ! sinking velocity (<0 for sinking) [m/d]
   wsed_spm       , & ! sedimentation rate [m/s]
   sediment_to    , & ! name of 2d tracer to which sedimentation takes place   
! settled matter parameters
   name_sed            , & ! name of sed tracer
   longname_sed        , & ! long name for output
   molar_volume_sed    , & ! specific volume of this sediment type [m3/mol]
   critical_stress_sed , & ! critical shear stress when erosion starts [N/m2]
   erosion_rate_sed    , & ! erosion rate [1/d]
   bioerosion_rate_sed , & ! erosion rate by benthic animals [1/d]
   suspend_to          , & ! name of 3d tracer to which resuspension  takes place
! sediment settings parameters
   sed_layer_height       , & ! maximum height of vertical layers [m]. <0 mean the layer may become infinitely thick.
   sed_layer_propagation  , & ! Sediment layer propagation settings, SLP_DOWNWARD=1, SLP_FULL_BOX=2, SLP_OLD_ERGOM=3
   sed_erosion_mode       , & ! Sediment erosion mode, INDEPENDENT=1, MAXSTRESS=2, ORGANIC=3
! other parameters
   vlev_sed               , & ! number of 2d tracers that may be stored in one diagnostic 3d tracer
                              ! set this parameter <= nk
   gamma_1                , & ! constant for opacity contribution of chlorophyll [kg/g/m]
   genus_style_par            ! calculate light intensity as it was done in the GENUS model

contains

  subroutine generic_ERGOM_register(tracer_list)
    type(g_tracer_type), pointer :: tracer_list

    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_register'
    character(len=fm_string_len) :: errorstring
    integer :: ioun, io_status, ierr, i, j
    logical :: found

    ! provide for namelist over-ride of defaults 
    ioun = open_namelist_file()
    read  (ioun, ergom_nml,iostat=io_status)
    write (stdout(),'(/)')
    write (stdout(), ergom_nml)
    write (stdlog(), ergom_nml)
    ierr = check_nml_error(io_status,'ergom_nml')
    call close_file (ioun)

    allocate(array_spm    (NUM_SPM)    )
    allocate(array_sed    (NUM_SED)    )
    allocate(sed_defs%layer_height(NUM_SEDIMENT_LAYERS))

    ! now, set the indexes for spm variables referred to
    ! A ) Set the values in sed
    do i=1,NUM_SED
      ! search the index of the spm variables to which resuspension takes place
      found = .false.
      array_sed(i)%index_suspend_to = -1
      do j=1,NUM_SPM
	if (trim(adjustl(suspend_to(i))) .eq. trim(adjustl(name_spm(j)))) then
	  found = .true.
	  array_sed(i)%index_suspend_to = j
	endif
      enddo
      if (.not. found) then
	if ((trim(adjustl(suspend_to(i))) .eq. 'none') .and. (NUM_SPM .eq. NUM_SED)) then
	  array_sed(i)%index_suspend_to = i
	elseif (trim(adjustl(suspend_to(i))) .ne. 'none') then
          write(errorstring, '(a)') &
		'Error: settled matter tracer '// &
		 trim(adjustl(name_sed(i)))	// &
		 ' shall be resuspended to tracer '	// & 
		 trim(adjustl(suspend_to(i))) // &
		 ', but that does not exist as an spm tracer.'
	  call  mpp_error(FATAL, errorstring)
	endif
      endif
    enddo
    
    ! now, set the indexes for sed variables referred to
    ! A) Set the values in spm
    do i=1,NUM_SPM
      ! search the index of the sed variables to which sedimentation takes place
      found = .false.
      array_spm(i)%index_sediment_to = -1
      do j=1,NUM_SED
	if (trim(adjustl(sediment_to(i))) .eq. trim(adjustl(name_sed(j)))) then
	  found = .true.
	  array_spm(i)%index_sediment_to = j
	endif
      enddo
      if (.not. found) then
	if ((trim(adjustl(sediment_to(i))) .eq. 'none') .and. (NUM_SPM .eq. NUM_SED)) then
	  array_spm(i)%index_sediment_to = i
	elseif (trim(adjustl(sediment_to(i))) .ne. 'none') then
          write(errorstring, '(a)') &
               'Error: suspended particulate matter (spm) tracer '// &
		trim(adjustl(name_spm(i)))     // &
		' shall be sedimented to tracer '     // & 
		trim(adjustl(sediment_to(i))) // &
		', but that does not exist as a settled matter (sed) tracer.'
	  call  mpp_error(FATAL, errorstring)
	endif
      endif
    enddo
    
    !Specify all prognostic and diagnostic tracers of this modules.
    call user_add_tracers(tracer_list)
    
  end subroutine generic_ERGOM_register

  ! <SUBROUTINE NAME="generic_ERGOM_init">
  !  <OVERVIEW>
  !   Initialize the generic ERGOM module
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   This subroutine: 
  !       Adds all the CFC Tracers to the list of generic Tracers passed to it 
  !       via utility subroutine g_tracer_add().
  !       Adds all the parameters used by this module via utility subroutine g_tracer_add_param().
  !       Allocates all work arrays used in the module. 
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_init(tracer_list)
  !  </TEMPLATE>
  !  <IN NAME="tracer_list" TYPE="type(g_tracer_type), pointer">
  !   Pointer to the head of generic tracer list.
  !  </IN>
  ! </SUBROUTINE>

  subroutine generic_ERGOM_init(tracer_list)
    type(g_tracer_type), pointer :: tracer_list

    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_init'
    integer :: ioun, io_status, ierr
    
    id_init     = mpp_clock_id('(ERGOM init) '            ,grain=CLOCK_ROUTINE)
    id_alloc    = mpp_clock_id('(ERGOM allocate) '        ,grain=CLOCK_ROUTINE)
    id_source   = mpp_clock_id('(ERGOM source) '          ,grain=CLOCK_ROUTINE)
    id_susp     = mpp_clock_id('(ERGOM resuspension) '    ,grain=CLOCK_ROUTINE)
    id_preloop  = mpp_clock_id('(ERGOM source pre-loop) ' ,grain=CLOCK_ROUTINE)
    id_mainloop = mpp_clock_id('(ERGOM source main loop) ',grain=CLOCK_ROUTINE)
    id_vertmig  = mpp_clock_id('(ERGOM source vmove) '    ,grain=CLOCK_ROUTINE)
    id_output   = mpp_clock_id('(ERGOM source output) '   ,grain=CLOCK_ROUTINE)

    call mpp_clock_begin(id_init)

    ! provide for namelist over-ride of defaults 
    ioun = open_namelist_file()
    read  (ioun, ergom_nml,iostat=io_status)
    write (stdout(),'(/)')
    write (stdout(), ergom_nml)
    write (stdlog(), ergom_nml)
    ierr = check_nml_error(io_status,'ergom_nml')
    call close_file (ioun)

    !Specify and initialize all parameters used by this package
    call user_add_params

    !Allocate and initiate all the private work arrays used by this module.
    call user_allocate_arrays 
    
    ! now print a summary of all parameters    
    write (stdout(),'(/)')
    write (stdout(),*) 'Summary of the ERGOM model setup'
    write (stdout(),*) 'CONSTANTS:'
<constants>
    write (stdout(),'((a), e13.6)')'    <name>		        : ', <name>
    write (stdout(),'(a)')         '        = <description>'    
</constants>
    write (stdout(),'(/)')
    write (stdout(),*) 'TRACERS (3d, no vertical movement):'
<tracers vertLoc=WAT; vertSpeed=0>
    write (stdout(),'(a)')         '    <name>'    
    write (stdout(),'((a), e13.6)')'        opacity        [m2/mol]: ',<opacity>
</tracers>
    write (stdout(),'(/)')
    write (stdout(),*) 'TRACERS (3d, vertical movement):'
<tracers vertLoc=WAT; vertSpeed/=0>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
    write (stdout(),'((a), e13.6)')'        vertical speed [m /day]: '//'<vertSpeed>=',<vertSpeedValue>
    write (stdout(),'((a), e13.6)')'        opacity        [m2/mol]: ',<opacity>
</tracers>
    write (stdout(),'(/)')
    write (stdout(),*) 'TRACERS (2d, at the surface):'
<tracers vertLoc=SUR>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
    write (stdout(),'((a), e13.6)')'        opacity        [m2/mol]: ',<opacity>
</tracers>
    write (stdout(),'(/)')
    write (stdout(),*) 'TRACERS (2d, at the bottom):'
<tracers vertLoc=SED>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
</tracers>
    write (stdout(),'(/)')
write (stdout(),*) 'TRACERS (pseudo-3d (fish)):'
<tracers vertLoc=FIS>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
</tracers>
    write (stdout(),'(/)')
    write (stdout(),*) 'PROCESSES in the surface layer:'
<processes vertLoc=SUR>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
</processes>
    write (stdout(),'(/)')
    write (stdout(),*) 'PROCESSES in the water column:'
<processes vertLoc=WAT>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
</processes>
    write (stdout(),'(/)')
    write (stdout(),*) 'PROCESSES in the bottom layer:'
<processes vertLoc=SED>
    write (stdout(),'(a)')         '    <name>  (<description>)'    
</processes>


    write (stdout(),'(a,I2)') 'Number of SPM types           : ', NUM_SPM
    write (stdout(),'(a,I2)') 'Number of settled matter types: ', NUM_SED
    do n=1, NUM_SPM
       write (stdout(),'(a)')            trim(array_spm(n)%name)//':'
       write (stdout(),'((a), e13.6)')  '  sinking velocity, wsink0 [m/d]               : ', array_spm(n)%wsink0  
       write (stdout(),'((a), e13.6)')  '  sedimentation rate, wsed [m/d]               : ', array_spm(n)%wsed    
       write (stdout(),'((a), (a))')    '  will sediment to tracer, (sediment_to)       : ', trim(array_sed(array_spm(n)%index_sediment_to)%name) 
    enddo
    do n=1, NUM_SED
       write (stdout(),'(a)')            trim(array_sed(n)%name)//':'
       write (stdout(),'((a), e13.6)')  '  critical shear stress, critical_stress [N/m2]: ', array_sed(n)%critical_stress  
       write (stdout(),'((a), (a))')    '  will be resuspended to tracer, (suspend_to)  : ', trim(array_spm(array_sed(n)%index_suspend_to)%name) 
    enddo
    write (stdout(),'(a,I2)') 'Number of vertical movement steps per timestep: ', NUM_VMOVE_STEPS
    write (stdout(),'(/)')

    call mpp_clock_end(id_init)

  end subroutine generic_ERGOM_init

  subroutine user_allocate_arrays
    integer :: isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau, n, i
    character(len=fm_string_len) :: mystring

    call mpp_clock_begin(id_alloc)

    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau) 
    !Allocate all the private arrays.

<tracers solubility/=0>
    allocate(ergom%<gasName>_alpha(isd:ied,jsd:jed)); ergom%<gasName>_alpha=0.0
    allocate(ergom%<gasName>_csurf(isd:ied,jsd:jed)); ergom%<gasName>_csurf=0.0
    allocate(ergom%<gasName>_sc_no(isd:ied,jsd:jed)); ergom%<gasName>_sc_no=0.0
</tracers>

    do n = 1, NUM_SPM
      allocate(array_spm(n)%move     (isd:ied,jsd:jed,nk))  ; array_spm(n)%move = 0.0
      allocate(array_spm(n)%diff     (isd:ied,jsd:jed,nk))  ; array_spm(n)%diff = 0.0
      allocate(array_spm(n)%btf      (isd:ied,jsd:jed))     ; array_spm(n)%btf  = 0.0
      allocate(array_spm(n)%jsed     (isd:ied,jsd:jed))     ; array_spm(n)%jsed = 0.0
    enddo
    do n = 1, NUM_SED
      allocate(array_sed(n)%f_sed    (isd:ied,jsd:jed,NUM_SEDIMENT_LAYERS)); array_sed(n)%f_sed = 0.0
      allocate(array_sed(n)%jgain_sed(isd:ied,jsd:jed)); array_sed(n)%jgain_sed = 0.0
      allocate(array_sed(n)%jloss_sed(isd:ied,jsd:jed)); array_sed(n)%jloss_sed = 0.0
      allocate(array_sed(n)%jres     (isd:ied,jsd:jed)); array_sed(n)%jres      = 0.0
      allocate(array_sed(n)%jbiores  (isd:ied,jsd:jed)); array_sed(n)%jbiores   = 0.0
      do i=1,NUM_SEDIMENT_LAYERS
	   write( mystring, '(i4)' )  i
	   call user_2d_tracer_assign_array(trim(array_sed(n)%name)//'_'//trim(adjustl(mystring)), &
    					 array_sed(n)%f_sed(:,:,i))
      enddo
    enddo
<auxiliaries isZIntegral=1>
    allocate(ergom%zintegralarray_<name>(isd:ied,jsd:jed)); ergom%zintegralarray_<name>=0.0
</auxiliaries>
<tracers vertLoc=FIS>
    allocate(ergom%cumulated_change_of_<name>(isd:ied,jsd:jed)); ergom%cumulated_change_of_<name>=0.0
</tracers>
    allocate(ergom%host_biores(isd:ied,jsd:jed)); ergom%host_biores  =0.0
    allocate(ergom%irr_inst(isd:ied,jsd:jed,nk)); ergom%irr_inst=0.0
    allocate(ergom%irr_zw(isd:ied,jsd:jed,nk)); ergom%irr_zw=0.0
    allocate(ergom%bio_opacity(isd:ied,jsd:jed,nk)); ergom%bio_opacity=0.0
<auxiliaries vertLoc=WAT; isUsedElsewhere=1>
    allocate(ergom%<name>(isd:ied,jsd:jed,nk)); ergom%<name>=0.0
</auxiliaries>
<celements isAging/=0>
    allocate(ergom%old_<aged>(isd:ied,jsd:jed,nk)); ergom%old_<aged>=0.0
</celements>
    
    call mpp_clock_end(id_alloc)

  end subroutine user_allocate_arrays

  subroutine generic_ERGOM_register_diag(diag_list)
    type(g_diag_type), pointer :: diag_list
    type(vardesc)  :: vardesc_temp
    integer        :: isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau, axes(3)
    type(time_type):: init_time 

    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau,axes=axes,init_time=init_time) 

<processes vertLoc=WAT>
    vardesc_temp = vardesc("<name>","<description>",'h','1','s','mol kg-1 s-1','f')
    ergom%id_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:3),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    if (ergom%id_<name> .gt. 0) then
      allocate(ergom%<name>(isd:ied,jsd:jed,nk))
      ergom%<name> = 0.0
    endif
</processes>

<processes vertLoc/=WAT>
    vardesc_temp = vardesc("<name>","<description>",'h','1','s','mol m-2 s-1','f')
    ergom%id_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    if (ergom%id_<name> .gt. 0) then
      allocate(ergom%<name>(isd:ied,jsd:jed))
      ergom%<name> = 0.0
    endif
</processes>


<auxiliaries vertLoc=WAT; isUsedElsewhere=0>
    vardesc_temp = vardesc("<name>","<description>",'h','1','s','1','f')
    ergom%id_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:3),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    if (ergom%id_<name> .gt. 0) then
      allocate(ergom%<name>(isd:ied,jsd:jed,nk))
      ergom%<name> = 0.0
    endif
</auxiliaries>

<auxiliaries vertLoc/=WAT>
    vardesc_temp = vardesc("<name>","<description>",'h','1','s','1','f')
    ergom%id_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    if (ergom%id_<name> .gt. 0) then
      allocate(ergom%<name>(isd:ied,jsd:jed))
      ergom%<name> = 0.0
    endif
</auxiliaries>

<tracers vertLoc=WAT; atmosDep/=0>
    vardesc_temp = vardesc("dep_wet_<name>","Atmospheric deposition of <description> to the ocean",'h','1','s','mol m-2 s-1','f')
    ergom%id_dep_wet_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)

</tracers>
<tracers vertLoc=WAT; riverDep/=0>
    vardesc_temp = vardesc("runoff_flux_<name>","<description> runoff flux to the ocean",'h','1','s','mol m-2 s-1','f')
    ergom%id_runoff_flux_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    vardesc_temp = vardesc("diffusive_flux_<name>","<description> diffusive flux from land to ocean",'h','1','s','mol m-2 s-1','f')
    ergom%id_diffusive_flux_<name> = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)

</tracers>

    vardesc_temp = vardesc("irr_inst","Instantaneous Light",'h','L','s','W m-2','f')
    ergom%id_irr_inst = register_diag_field(package_name, vardesc_temp%name, axes(1:3),&
         init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)

    do n=1, NUM_SPM
        vardesc_temp = vardesc(trim(array_spm(n)%name)//"_jsed", &
               "Sedimentation of "//trim(array_spm(n)%longname),'h','L','s','mol m-2 s-1','f')
        array_spm(n)%id_jsed = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
          init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    enddo

    do n=1, NUM_SED
        vardesc_temp = vardesc(trim(array_sed(n)%name)//"_jgain_sed", &
               "Gain of "//trim(array_sed(n)%longname)//" by transformation of other sediment classes", &
	               'h','L','s','mol m-2 s-1','f')
        array_sed(n)%id_jgain_sed = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
          init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
        vardesc_temp = vardesc(trim(array_sed(n)%name)//"_jloss_sed", &
               "Loss of "//trim(array_sed(n)%longname)//" by transformation of sediment",'h','L','s','mol m-2 s-1','f')
        array_sed(n)%id_jloss_sed = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
          init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
        vardesc_temp = vardesc(trim(array_sed(n)%name)//"_jres", &
               "Resuspension of "//trim(array_sed(n)%longname),'h','L','s','mol m-2 s-1','f')
        array_sed(n)%id_jres = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
          init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
        vardesc_temp = vardesc(trim(array_sed(n)%name)//"_jbiores", &
               "resuspension of "//trim(array_sed(n)%longname)//" by benthic organisms",'h','L','s','mol m-2 s-1','f')
        array_sed(n)%id_jbiores = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
          init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    enddo

    call user_register_2d_tracers

  end subroutine generic_ERGOM_register_diag

  !
  !   This is an internal sub, not a public interface.
  !   Add all the parameters to be used in this module. 
  !
  subroutine user_add_params
  
  !Specify all parameters used in this modules.
  !=============================================================================
  
  !Add the known experimental parameters used for calculations
  !in this module.
  !All the g_tracer_add_param calls must happen between 
  !g_tracer_start_param_list and g_tracer_end_param_list  calls.
  !This implementation enables runtime overwrite via field_table.
  
    integer                      :: n
    character(len=fm_string_len) :: mystring
    call g_tracer_start_param_list(package_name)

    !-----------------------------------------------------------------------
    ! NITROGEN
    !-----------------------------------------------------------------------    
    !  Schmidt number coefficients [1/psu]
    !-----------------------------------------------------------------------
    call g_tracer_add_param('a1_nit', ergom%a1_nit, 2206.1)
    call g_tracer_add_param('a2_nit', ergom%a2_nit, -144.86)
    call g_tracer_add_param('a3_nit', ergom%a3_nit, 4.5413)
    call g_tracer_add_param('a4_nit', ergom%a4_nit, -0.056988)
    !-----------------------------------------------------------------------
    !     Solubility coefficients for (alpha in mol/kg/atm)
    !     Hamme and Emerson, 2004
    !     Hamme, R.C. and Emerson, S.R., 2004. The solubility of neon, nitrogen and argon 
    !     in distilled water and seawater. 
    !     Deep Sea Research Part I: Oceanographic Research Papers, 51(11): 1517-1528.
    !-----------------------------------------------------------------------
    call g_tracer_add_param('RHO_0',  ergom%Rho_0, 1035.0)
    call g_tracer_add_param('t1_nit', ergom%t1_nit, 6.42931)
    call g_tracer_add_param('t2_nit', ergom%t2_nit, 2.92704)
    call g_tracer_add_param('t3_nit', ergom%t3_nit, 4.32531)
    call g_tracer_add_param('t4_nit', ergom%t4_nit, 4.69149)
    call g_tracer_add_param('e1_nit', ergom%e1_nit, 298.15)
    call g_tracer_add_param('s1_nit', ergom%s1_nit, 7.44129e-3)
    call g_tracer_add_param('s2_nit', ergom%s2_nit, 8.02566e-3)
    call g_tracer_add_param('s3_nit', ergom%s3_nit, 0.0146775)
    
    !---------------------------------------------------------------------
    ! OXYGEN
    !---------------------------------------------------------------------
    !  Compute the Schmidt number of O2 in seawater using the 
    !  formulation proposed by Keeling et al. (1998, Global Biogeochem.
    !  Cycles, 12, 141-163).
    !---------------------------------------------------------------------
    call g_tracer_add_param('a1_o2',  ergom%a1_o2, 1929.7)
    call g_tracer_add_param('a2_o2',  ergom%a2_o2, -117.46)
    call g_tracer_add_param('a3_o2',  ergom%a3_o2, 3.116)
    call g_tracer_add_param('a4_o2',  ergom%a4_o2, -0.0306)
    !-----------------------------------------------------------------------
    !     Solubility coefficients for alpha in mol/kg/atm FOR OXYGEN
    !-----------------------------------------------------------------------
    call g_tracer_add_param('t0_o2',  ergom%t0_o2, 2.00907)
    call g_tracer_add_param('t1_o2',  ergom%t1_o2, 3.22014)
    call g_tracer_add_param('t2_o2',  ergom%t2_o2, 4.05010)
    call g_tracer_add_param('t3_o2',  ergom%t3_o2, 4.94457)
    call g_tracer_add_param('t4_o2',  ergom%t4_o2, -2.56847e-01)
    call g_tracer_add_param('t5_o2',  ergom%t5_o2, 3.88767)
    call g_tracer_add_param('b0_o2',  ergom%b0_o2, -6.24523e-03)
    call g_tracer_add_param('b1_o2',  ergom%b1_o2, -7.37614e-03)
    call g_tracer_add_param('b2_o2',  ergom%b2_o2, -1.03410e-02 )
    call g_tracer_add_param('b3_o2',  ergom%b3_o2, -8.17083e-03)
    call g_tracer_add_param('c0_o2',  ergom%c0_o2, -4.88682e-07)

    !---------------------------------------------------------------------    
    ! CARBON DIOXIDE
    !---------------------------------------------------------------------
    !  Compute the Schmidt number of CO2 in seawater using the 
    !  formulation presented by Wanninkhof (1992, J. Geophys. Res., 97,
    !  7373-7382).
    !-----------------------------------------------------------------------
    call g_tracer_add_param('a1_co2', ergom%a1_co2,  2068.9)
    call g_tracer_add_param('a2_co2', ergom%a2_co2, -118.63)
    call g_tracer_add_param('a3_co2', ergom%a3_co2,  2.9311)
    call g_tracer_add_param('a4_co2', ergom%a4_co2, -0.027)

    ! suspended particulate matter parameters
    do n=1, NUM_SPM
        call g_tracer_add_param(trim(name_spm(n))//'name',        array_spm(n)%name,        name_spm(n))      
        call g_tracer_add_param(trim(name_spm(n))//'wsink0',      array_spm(n)%wsink0,      wsink0_spm(n)/sperd) 
        call g_tracer_add_param(trim(name_spm(n))//'wsed',        array_spm(n)%wsed,        wsed_spm(n)/sperd)   
        call g_tracer_add_param(trim(name_spm(n))//'sediment_to', array_spm(n)%sediment_to, sediment_to(n))  
    enddo
    ! settled matter parameters
    do n=1, NUM_SED
        call g_tracer_add_param(trim(name_sed(n))//'name_2d',     array_sed(n)%name,        name_sed(n))      
        call g_tracer_add_param(trim(name_sed(n))//'suspend_to',  array_sed(n)%suspend_to,  suspend_to(n))   
        call g_tracer_add_param(trim(name_sed(n))//'erosion_rate',array_sed(n)%erosion_rate,& 
	                             erosion_rate_sed(n)/sperd) 
        call g_tracer_add_param(trim(name_sed(n))//'bioerosion_rate',array_sed(n)%bioerosion_rate, &
	                             bioerosion_rate_sed(n)/sperd) 
        call g_tracer_add_param(trim(name_sed(n))//'molar_volume',array_sed(n)%molar_volume,molar_volume_sed(n)) 
        call g_tracer_add_param(trim(name_sed(n))//'critical_stress',array_sed(n)%critical_stress, &
	                             critical_stress_sed(n))
      call g_tracer_add_param(trim(longname_sed(n))//'longname',   array_sed(n)%longname,    longname_sed(n))       
    enddo

    ! sediment settings parameters
    do n=1,NUM_SEDIMENT_LAYERS
      write( mystring, '(i4)' )  n
      ! (maximum) height of vertical layers [m]. Numbers <0 mean the layer may become infinitely thick.
      call g_tracer_add_param('sed_layer_height_'//trim(adjustl(mystring)), sed_defs%layer_height(n), & 
                                     sed_layer_height(n))
    enddo
    call g_tracer_add_param('sed_layer_propagation', sed_defs%layer_propagation, sed_layer_propagation) 
    call g_tracer_add_param('sed_erosion_mode', sed_defs%erosion_mode     , sed_erosion_mode     ) 
    call g_tracer_add_param('NUM_SEDIMENT_LAYERS'  , sed_defs%NUM_LAYERS       , NUM_SEDIMENT_LAYERS	   ) 

    call g_tracer_add_param('gamma_1'  , ergom%gamma_1       , gamma_1   ) 
    call g_tracer_add_param('RHO_0'    , ergom%rho_0         , 1035.0   )     

    call g_tracer_end_param_list()
    
  end subroutine user_add_params

  !
  !   This is an internal sub, not a public interface.
  !   Add all the tracers to be used in this module. 
  !
  subroutine user_add_tracers(tracer_list)
    type(g_tracer_type), pointer :: tracer_list

    character(len=fm_string_len), parameter :: sub_name = 'user_add_tracers'
    integer                                 :: i, n
    character(len=fm_string_len)            :: mystring

    call g_tracer_start_param_list(package_name)
    call g_tracer_add_param('ice_restart_file'   , ergom%ice_restart_file   , 'ice_ergom.res.nc')
    call g_tracer_add_param('ocean_restart_file' , ergom%ocean_restart_file , 'ocean_ergom.res.nc' )
    call g_tracer_add_param('IC_file'            , ergom%IC_file       , '')
    call g_tracer_end_param_list()

    ! Set Restart files
    call g_tracer_set_files(ice_restart_file=ergom%ice_restart_file, ocean_restart_file=ergom%ocean_restart_file )

    !=====================================================
    !Specify all prognostic tracers of this modules.
    !=====================================================
    !User adds one call for each prognostic tracer below!
    !User should specify if fluxes must be extracted from boundary 
    !by passing one or more of the following methods as .true.  
    !and provide the corresponding parameters array
    !methods: flux_gas,flux_runoff,flux_wetdep,flux_drydep  
    !
    !prog_tracers: nitrogen 
    !diag_tracers: none
    !

    call user_init_2d_tracer_list

<backwardTracers vertLoc=WAT; vertSpeed/=0>
    call g_tracer_add(tracer_list,package_name,&
       name       = '<name>',          &
       longname   = '<description>',   &
  <if useInitValue/=0>
    <if initValue=0.0>
       const_init_value = 1.0e-20,     &
    </if>
    <if initValue/=0.0>
       const_init_value = <initValue>, &
    </if>
  </if>
       flux_runoff= (<riverDep> > 0),  &
       flux_gas   = .false.,           &
       flux_wetdep= (<atmosDep> > 0),  &
       flux_drydep= (<riverDep> > 0),  &
       flux_param = (/ 1.0 /),         &
       units      = 'mol/kg',          &
       prog       = .true. ,           &
       flux_bottom= .true. ,           &
       move_vertical = .true. ,        &
       diff_vertical = .true.)
</tracers>
    do n=1, NUM_SED
        do i=1,NUM_SEDIMENT_LAYERS
          write( mystring, '(i4)' )  i
          call user_add_2d_tracer(tracer_list,    &
             name       = trim(name_sed(n))//'_'//trim(adjustl(mystring)),  &
             longname   = trim(longname_sed(n))//' concentration in sediment layer '//trim(adjustl(mystring)), &
             units      = 'mol/m^2')
        enddo
    enddo
    
<backwardTracers vertLoc=WAT; vertSpeed=0>
  <if solubility/=0>
    ! a comment to "flux_gas_param":
    ! The first term of flux_gas_param [s/m] describes the relation between (10m wind speed)**2 and gas transfer velocity:
    ! flux       = (csat-c) * sqrt(660/schmidt_number) * param1 * u10**2
    ! [mol/m2/s] = [mol/m3] * [1]                      * [s/m]  * [m2/s2]
    ! param1 = 8.61e-7 according to eqn. 3 in: Wannikhof (1992): Relationship between Wind Speed and Gas Exchange over the Ocean, JGR, 1997,
    ! Wannikhof (1992): Relationship between Wind Speed and Gas Exchange over the Ocean, JGR, Vol. 97, page 7376
  </if>
    call g_tracer_add(tracer_list,package_name,&
         name       = '<name>',    &
         longname   = '<description>',      &
  <if useInitValue/=0>
    <if initValue=0.0>
         const_init_value = 1.0e-20,     &
    </if>
    <if initValue/=0.0>
         const_init_value = <initValue>, &
    </if>
  </if>
         units      = 'mol/kg',          &
         prog       = .true.,            &
  <if solubility=0>
         flux_gas   = .false.,           &
  </if>
  <if solubility/=0>
         flux_gas   = .true.,            &
         flux_gas_name  = '<gasName>_flux',    &
         flux_gas_type  = 'air_sea_gas_flux_generic',&
         flux_gas_param = (/ 9.36e-07, 1.0 /), & 
         flux_gas_restart_file  = 'ocean_ergom_airsea_flux.res.nc', &
         flux_gas_molwt = <molarMass>, &           
  </if>
         flux_bottom= .false.,           &
         flux_runoff= (<riverDep> > 0),  &
         flux_param = (/ 1.0 /),         &
         flux_wetdep= (<atmosDep> > 0),  &
         flux_drydep= (<riverDep> > 0),  &
         btm_reservoir=.false. )
</backwardTracers>

<auxiliaries vertLoc=WAT; isUsedElsewhere=1>
   call g_tracer_add(tracer_list,package_name,&
         name       = '<name>', &
         longname   = '<description>', &
         units      = '1',     &
         prog       = .false.,       &
         const_init_value = 1.0e-20           )

</auxiliaries>

   call g_tracer_add(tracer_list,package_name,&
         name       = 'chl',         &
         longname   = 'Chlorophyll', &
         units      = 'ug kg-1',     &
         prog       = .false.,       &
         const_init_value = 0.08           )

    call g_tracer_add(tracer_list,package_name,&
         name       = 'opacity_bio', &
         longname   = 'Opacity of water ingredients', &
         units      = 'm-1',     &
         prog       = .false.,       &
         const_init_value = 1.0e-20           )
	 
    call g_tracer_add(tracer_list,package_name,&
         name       = 'opacity_water', &
         longname   = 'Opacity of pure water', &
         units      = 'm-1',     &
         prog       = .false.,       &
         const_init_value = 0.18           )

  end subroutine user_add_tracers

  !
  !   This is an internal sub, not a public interface.
  !   initialize the list of 2d tracers with a length of zero
  !
  subroutine user_init_2d_tracer_list
    character(len=fm_string_len), parameter :: sub_name = 'user_init_2d_tracer_list'
    allocate(tracers_2d(0))
  end subroutine user_init_2d_tracer_list
  !
  !   This is an internal sub, not a public interface.
  !   Insert a 2d tracer into the 2d tracer list
  !   2d tracers are stored  together in diagnostic 3d tracers.
  !   Those are registered for output and in the restart list.
  !
  subroutine user_add_2d_tracer(tracer_list,name,longname,units)
    type(g_tracer_type), pointer :: tracer_list
    character(len=*), intent(in) :: name, longname, units

    character(len=fm_string_len), parameter :: sub_name = 'user_add_2d_tracer'
    integer                      :: m, n, dummy
    character(len=fm_string_len) :: temp_string
    type(tracer_2d), ALLOCATABLE, dimension(:) :: temp_tracers_2d
    
    n  = size(tracers_2d)
    allocate(temp_tracers_2d(n))
    do m=1,n
      temp_tracers_2d(m)=tracers_2d(m)
    end do
    deallocate(tracers_2d)
    allocate(tracers_2d(n+1))
    do m=1,n
      tracers_2d(m)=temp_tracers_2d(m)
    end do
    deallocate(temp_tracers_2d)
    
    tracers_2d(n+1)%name     = trim(adjustl(name))
    tracers_2d(n+1)%longname = trim(adjustl(longname))
    tracers_2d(n+1)%units    = trim(adjustl(units))
    tracers_2d(n+1)%layer_in_3d_tracer = mod(n,vlev_sed)+1
    tracers_2d(n+1)%field_assigned     = .false.  ! whether a 2d field has been assigned
    
    m = n/vlev_sed+1
    write( temp_string, '(i4)' )  m; temp_string = adjustl(temp_string)
    tracers_2d(n+1)%name_of_3d_tracer = 'tracer_2d_'//trim(temp_string)
    if (mod(n,vlev_sed) .eq. 0) then
      call g_tracer_add(tracer_list,package_name,                              &
         name       = 'tracer_2d_'//trim(temp_string),                         &
         longname   = 'Tracer '//trim(temp_string)//' containing 2d variables',&
         units      = 'none',                                                  &
         prog       = .false.)
    endif
  end subroutine user_add_2d_tracer

  !   This is an internal sub, not a public interface.
  !   Register the 2d tracer for output via a 2d output field
  subroutine user_register_2d_tracers
    real,parameter :: missing_value1=-1.0e+20
    type(vardesc)  :: vardesc_temp
    integer        :: isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau, axes(3)
    type(time_type):: init_time 
    integer        :: m,n

    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau,axes=axes,init_time=init_time) 

    n=size(tracers_2d)
    do m=1,n
      vardesc_temp = vardesc(                                                                   &
      tracers_2d(m)%name, tracers_2d(m)%longname,'h','L','s',tracers_2d(m)%units,'f')
      tracers_2d(m)%diag_field = register_diag_field(package_name, vardesc_temp%name, axes(1:2),&
	      init_time, vardesc_temp%longname,vardesc_temp%units, missing_value = missing_value1)
    end do
  end subroutine user_register_2d_tracers

  !   This is an internal sub, not a public interface.
  !   Let the p_field pointer of the 2d tracer point to the data array (for read and write)
  subroutine user_2d_tracer_assign_array(name,array)
    character(len=*), intent(in)           :: name
    real,dimension(:,:),target             :: array
    integer                                :: n
    logical                                :: found_tracer
    character(len=fm_string_len)           :: errorstring

    found_tracer=.false.
    do n=1,size(tracers_2d)
      if (trim(adjustl(name)) .eq. trim(adjustl(tracers_2d(n)%name))) then
        tracers_2d(n)%p_field => array
        tracers_2d(n)%field_assigned = .true.
        found_tracer=.true.
      end if
    end do
    if (.not. found_tracer) then
       write(errorstring, '(a)') &
       'array assigned to tracer '//trim(name)// &
       ', but that tracer was not added by user_add_2d_tracer'
       call  mpp_error(FATAL, errorstring)
    end if
  end subroutine user_2d_tracer_assign_array

  !   This is an internal sub, not a public interface.
  !   Load all data stored in the (3d tracers containing 2d tracer data) into their temporary 2d arrays
   subroutine user_get_2d_tracer_values(tracer_list,isd,ied,jsd,jed,nk)
    type(g_tracer_type),    pointer      :: tracer_list
    integer,                  intent(in) :: isd,ied,jsd,jed,nk
    real,dimension(:,:,:),allocatable    :: a3d
    integer                              :: n
    character(len=fm_string_len)         :: loaded_3d_tracer
    allocate(a3d(isd:ied,jsd:jed,1:nk))
    loaded_3d_tracer=''
    do n=1,size(tracers_2d)
      if (tracers_2d(n)%field_assigned) then
        if (trim(adjustl(tracers_2d(n)%name_of_3d_tracer)) .ne. trim(adjustl(loaded_3d_tracer))) then
          call g_tracer_get_values(     &
	  tracer_list,tracers_2d(n)%name_of_3d_tracer,'field',a3d,isd,jsd,ntau=1,positive=.true.)
          loaded_3d_tracer=tracers_2d(n)%name_of_3d_tracer
        end if
        tracers_2d(n)%p_field = a3d(:,:,tracers_2d(n)%layer_in_3d_tracer)
      end if
    end do
    deallocate(a3d)
  end subroutine user_get_2d_tracer_values

  !   This is an internal sub, not a public interface.
  !   Save all 2d tracer data from their temporary 2d arrays into the 3d tracers containing 2d tracer data
  subroutine user_set_2d_tracer_values(tracer_list ,model_time)
    type(g_tracer_type),      pointer         :: tracer_list
    type(time_type),          intent(in)      :: model_time
    real, dimension(:,:,:),   pointer         :: grid_tmask
    
    integer                                   :: isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau
    real,dimension(:,:,:),allocatable         :: a3d
    integer                                   :: n, layer
    character(len=fm_string_len)              :: loaded_3d_tracer
    logical                                   :: used

    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau,grid_tmask=grid_tmask) 

    allocate(a3d(isd:ied,jsd:jed,1:nk))
    loaded_3d_tracer=''
    do n=1,size(tracers_2d)
      if (tracers_2d(n)%field_assigned) then
        if (trim(adjustl(tracers_2d(n)%name_of_3d_tracer)) .ne. trim(adjustl(loaded_3d_tracer))) then 
          ! a new 3d tracer starts -> first save the temporarily stored data into the old 3d tracer
          if (loaded_3d_tracer .ne. '') then
            call g_tracer_set_values(tracer_list,loaded_3d_tracer,'field',a3d,isd,jsd,ntau=1)
          end if
          ! then, nullify the temporary tracer
          a3d=0.0
          loaded_3d_tracer=tracers_2d(n)%name_of_3d_tracer
        end if
        ! save the data to the temporary tracer
        layer = tracers_2d(n)%layer_in_3d_tracer
        a3d(:,:,layer)  = tracers_2d(n)%p_field
        ! save the values to the diagnostic field
        used = send_data(tracers_2d(n)%diag_field, a3d(:,:,layer),      & 
                  model_time, rmask = grid_tmask(:,:,1), is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
      end if
    end do
    !now, save the last tracer
    if (loaded_3d_tracer .ne. '') then
      call g_tracer_set_values(tracer_list,loaded_3d_tracer,'field',a3d,isd,jsd,ntau=1)
    end if
    deallocate(a3d)
  end subroutine user_set_2d_tracer_values

  ! <SUBROUTINE NAME="generic_ERGOM_update_from_coupler">
  !  <OVERVIEW>
  !   Modify the values obtained from the coupler if necessary.
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Currently an empty stub for CFCs.
  !   Some tracer fields need to be modified after values are obtained from the coupler.
  !   This subroutine is the place for specific tracer manipulations.
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_update_from_coupler(tracer_list) 
  !  </TEMPLATE>
  !  <IN NAME="tracer_list" TYPE="type(g_tracer_type), pointer">
  !   Pointer to the head of generic tracer list.
  !  </IN>
  ! </SUBROUTINE>
  subroutine generic_ERGOM_update_from_coupler(tracer_list)
    type(g_tracer_type), pointer :: tracer_list
    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_update_from_coupler'
    !
    !Nothing specific to be done for CFC's
    !
    return
  end subroutine generic_ERGOM_update_from_coupler

  ! <SUBROUTINE NAME="sedimentation_and_resuspension">
  !  <OVERVIEW>
  !    Perform sedimentation and resuspension for all spm and sed tracers
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !    All tracers that are able to be sedimented are stored in the spm array.
  !    All tracers that are able to be resuspended are stored in the sed array.
  !    This subroutine performs the sedimentation and resuspension.
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call sedimentation_and_resuspension(NUM_SPM, spm, NUM_SED, sed,    &
  !isc, iec, jsc, jec, isd, ied, jsd, jed, grid_kmt, dzt, rho_dzt, tau, dt, &
  !sed_defs, current_wave_stress, bioerosion)
  !  </TEMPLATE>
  ! </SUBROUTINE>
  subroutine sedimentation_and_resuspension(NUM_SPM, spm, NUM_SED, sed,    &
  isc, iec, jsc, jec, isd, ied, jsd, jed, grid_kmt, dzt, rho_dzt, tau, dt, &
  sed_defs, current_wave_stress, bioerosion)
    use mpp_mod,                only: mpp_pe
    integer, intent(in)                                     :: NUM_SPM,        &
                                                               NUM_SED,        &
                                                               isc, iec, jsc, jec, &
							       isd, ied, jsd, jed, tau
    type(spm_type), intent(inout), dimension(:)             :: spm
    type(sed_type), intent(inout), dimension(:)             :: sed
    integer, dimension(isd:,jsd:)                           :: grid_kmt
    real, intent(in), dimension(isd:,jsd:,:)                :: dzt, rho_dzt
    real, intent(in)                                        :: dt
    type(sed_defs_type), intent(inout)                      :: sed_defs
    real, dimension(isd:,jsd:),     intent(in)              :: current_wave_stress
    real, dimension(isd:,jsd:), intent(in)                  :: bioerosion


    
    integer                            :: i,j,k,l,n,n_sed
    real                               :: temp1, diff_stress
    real,  allocatable, dimension(:,:) :: work1
    real,  parameter                   :: lbi = 0.007/24/3600   ! 7 promille per day in [1/s]
    real,  parameter                   :: ironpo4norm = 0.0045  ! [mol/m/m]
    real,  parameter                   :: ironpo4th   = 0.1     ! [mol/m/m]
    real,  parameter                   :: smax = 4.5            ! maximum nitrogen in sediment
    
    call mpp_clock_begin(id_susp)
    
    do n=1,NUM_SPM
       spm(n)%btf = 0
    enddo !n
   
    !sedimentation
    do n=1,NUM_SPM
       if (spm(n)%index_sediment_to .gt. 0) then
          n_sed=spm(n)%index_sediment_to
          do j = jsc, jec; do i = isc, iec
             k=grid_kmt(i,j)
             if (k == 0) cycle
             temp1 = spm(n)%p_wat(i,j,k,tau) * spm(n)%wsed * ergom%rho_0             ! settling rate [mol m-2 s-1]
             spm(n)%btf(i,j) = spm(n)%btf(i,j) + temp1
             sed(n_sed)%f_sed(i,j,1)   = &
          		sed(n_sed)%f_sed(i,j,1) + temp1  * dt		      ! add suspended matter to sediment
             if (spm(n)%id_jsed .gt. 0) spm(n)%jsed(i,j) = temp1	      ! output of sedimentation
          enddo; enddo !i,j
       endif
    enddo !n

    !resuspension
    selectcase(sed_defs%erosion_mode)
    case(INDEPENDENT)
      do n=1,NUM_SED
          if (sed(n)%index_suspend_to .gt. 0) then
             n_sed=sed(n)%index_suspend_to
             do j = jsc, jec; do i = isc, iec
                k=grid_kmt(i,j)
                if (k == 0) cycle
                if (current_wave_stress(i,j) .gt. sed(n)%critical_stress) then
! constant erosion independent off stress. The more organic matter is in the
! sediment the more will be suspended. Inorganic matter is not considered here.
        	   temp1 = sed(n)%f_sed(i,j,1) * sed(n)%erosion_rate	   ! erosion rate [mol m-2 s-1]
        	   if (sed(n)%id_jres .gt. 0) sed(n)%jres(i,j) = temp1	   ! output of resuspension
        	   sed(n)%f_sed(i,j,1) = sed(n)%f_sed(i,j,1) - temp1  * dt ! remove sediment [mol m-2]
                   spm(n_sed)%btf(i,j) = spm(n_sed)%btf(i,j) - temp1
                else
        	   if (sed(n)%id_jres .gt. 0) sed(n)%jres(i,j) = 0.
                endif
             enddo; enddo !i,j
          endif
      enddo !n

    case(ORGANIC)
      do n=1,NUM_SED
         if (sed(n)%index_suspend_to .gt. 0) then
            n_sed=sed(n)%index_suspend_to
            do j = jsc, jec; do i = isc, iec
              k=grid_kmt(i,j)
              if (k == 0) cycle
	      diff_stress = current_wave_stress(i,j) - sed(n)%critical_stress
              if (diff_stress .gt. 0.0) then
        	 temp1 = diff_stress * sed(n)%erosion_rate	           ! erosion rate [mol m-2 s-1]
        	 if (sed(n)%id_jres .gt. 0) sed(n)%jres(i,j) = temp1	   ! output of resuspension
        	 sed(n)%f_sed(i,j,1) = sed(n)%f_sed(i,j,1) - temp1  * dt   ! remove sediment [mol m-2]
                 spm(n_sed)%btf(i,j) = spm(n_sed)%btf(i,j) - temp1
!        	 spm(n_sed)%p_wat(i,j,k,tau)   = &
!        		   spm(n_sed)%p_wat(i,j,k,tau) + temp1  * dt	   ! add concentration to spm
               else
        	  if (sed(n)%id_jres .gt. 0) sed(n)%jres(i,j) = 0.
               endif
            enddo; enddo !i,j
         endif
      enddo !n
    case(MAXSTRESS)
    case DEFAULT
       call  mpp_error(FATAL, &
      '==>Error from ERGOM, undefined erosion mode.')
    end select

    !bioresuspension
    do n=1,NUM_SED
       if (sed(n)%index_suspend_to .gt. 0) then
          n_sed=sed(n)%index_suspend_to
          do j = jsc, jec; do i = isc, iec
             k=grid_kmt(i,j)
             if (k == 0) cycle
             temp1 = sed(n)%f_sed(i,j,1) * sed(n)%bioerosion_rate * bioerosion(i,j) ! bioerosion [mol m-2 s-1]
             if (sed(n)%id_jbiores .gt. 0) sed(n)%jbiores(i,j) = temp1  	    ! output of bioerosion
             sed(n)%f_sed(i,j,1) = sed(n)%f_sed(i,j,1) - temp1  * dt		    ! remove sediment [mol m-2]
             spm(n_sed)%btf(i,j) = spm(n_sed)%btf(i,j) - temp1
          enddo; enddo !i,j
       endif
    enddo !n


    !layer propagation
    selectcase(sed_defs%layer_propagation)
    case(SLP_DOWNWARD)
      !if the surface box is overfull, the same volume propagates downward through all boxes
      allocate(work1(isc:iec,jsc:jec))
      do l=1, sed_defs%NUM_LAYERS
         !calculate the volume in the box
         work1 = 0.0
         do n=1, NUM_SED
            do j = jsc, jec; do i = isc, iec
               k = grid_kmt(i,j)
               if (k == 0) cycle
               work1(i,j) = work1(i,j) + sed(n)%f_sed(i,j,1)*sed(n)%molar_volume ![(mol/m2) * (m3/mol) = m]
            enddo; enddo !i,j
         enddo !n
         !calculate the ratio between current and maximal box height
         do j = jsc, jec; do i = isc, iec
            k=grid_kmt(i,j)
            if (k == 0) cycle
            work1(i,j) = max(work1(i,j)/sed_defs%layer_height(l),1.0)  ! if layer_height<0, nothing is transferred
         enddo; enddo !i,j
         !divide the amount in the current box by this ratio
         do n=1, NUM_SED
            do j = jsc, jec; do i = isc, iec
               k=grid_kmt(i,j)
               if (k == 0) cycle
               sed(n)%f_sed(i,j,1) = sed(n)%f_sed(i,j,1)/work1(i,j) 
            enddo; enddo !i,j
         enddo !n
         !calculate which fraction needs to be transferred downward
         do j = jsc, jec; do i= isc, iec
            k=grid_kmt(i,j)
            if (k == 0) cycle
            work1(i,j) = work1(i,j) - 1.0 !e.g. 0.5
         enddo; enddo !i,j
         !store everything in the box below
         if (l .lt. sed_defs%NUM_LAYERS) then
            do n=1, NUM_SED
               do j = jsc, jec; do i = isc, iec
         	  k=grid_kmt(i,j)
         	  if (k == 0) cycle
         	  sed(n)%f_sed(i,j,l+1) = sed(n)%f_sed(i,j,l+1) + sed(n)%f_sed(i,j,l)*work1(i,j)
               enddo; enddo !i,j
            enddo !n
         endif
      enddo !l
      deallocate(work1)
    case(SLP_FULL_BOX)
    case(SLP_OLD_ERGOM)
      !if the surface box is overfull, sediment detritus propagates down through all layers. 
      allocate(work1(isc:iec,jsc:jec))
      do l=1, NUM_SEDIMENT_LAYERS
         !calculate the volume in the box
         work1=0.0
         do n=1, 1 !NUM_SED
            do j = jsc, jec; do i = isc, iec
               k=grid_kmt(i,j)
               if (k == 0) cycle
               work1(i,j) = work1(i,j) + sed(n)%f_sed(i,j,1)*sed(n)%molar_volume ![(mol/m2) * (m3/mol) = m]
            enddo; enddo !i,j
         enddo !n
         !calculate the ratio between current and maximal box height
         do j = jsc, jec; do i = isc, iec
            k=grid_kmt(i,j)
            if (k == 0) cycle
            work1(i,j) = max(work1(i,j)/sed_defs%layer_height(l),1.0)  ! if layer_height<0, nothing is transferred
         enddo; enddo !i,j
         !divide the amount in the current box by this ratio
         do n=1, 1 !NUM_SED
            do j = jsc, jec; do i = isc, iec
               k=grid_kmt(i,j)
               if (k == 0) cycle
               sed(n)%f_sed(i,j,1) = sed(n)%f_sed(i,j,1)/work1(i,j) 
            enddo; enddo !i,j
         enddo !n
         !calculate which fraction needs to be transferred downward
         do j = jsc, jec; do i = isc, iec
            k=grid_kmt(i,j)
            if (k == 0) cycle
            work1(i,j) = work1(i,j) - 1.0 !e.g. 0.5
         enddo; enddo !i,j
         !store everything in the box below
         if (l .lt. sed_defs%NUM_LAYERS) then
            do n=1,1 !NUM_SED
               do j = jsc, jec; do i = isc, iec
         	  k=grid_kmt(i,j)
         	  if (k == 0) cycle
         	  sed(n)%f_sed(i,j,l+1) = sed(n)%f_sed(i,j,l+1) + sed(n)%f_sed(i,j,l)*work1(i,j)
               enddo; enddo !i,j
            enddo !n
         endif
      enddo !l
      ! now, store some iron phosphate from the uppermost box in the second one
      do j = jsc, jec; do i = isc, iec
         k=grid_kmt(i,j)
         if (k == 0) cycle
         temp1 = sed(2)%f_sed(i,j,1)  ! iron phosphate concentration [mol/m2]
         temp1 = max(temp1,temp1+temp1*(temp1-ironpo4th)/ironpo4norm)*lbi*temp1/smax ! burial of iron phosphate [mol/m2/s]
         sed(2)%f_sed(i,j,1) = sed(2)%f_sed(i,j,1) - temp1*dt                        ! remove from first layer [mol/m2]
         sed(2)%f_sed(i,j,2) = sed(2)%f_sed(i,j,2) + temp1*dt                        ! store in second layer [mol/m2]
      enddo; enddo
      deallocate(work1)
    
    case DEFAULT
       call  mpp_error(FATAL, &
      '==>Error from ERGOM, undefined sediment layer propagation.')
    end select
    call mpp_clock_end(id_susp)

  end subroutine sedimentation_and_resuspension

  ! <SUBROUTINE NAME="generic_ERGOM_update_from_bottom">
  !  <OVERVIEW>
  !   Set values of bottom fluxes and reservoirs
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Some tracers have bottom fluxes and reservoirs. 
  !   This subroutine is the place for specific tracer manipulations.
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_update_from_bottom(tracer_list,dt, tau, model_time) 
  !  </TEMPLATE>
  !  <IN NAME="tracer_list" TYPE="type(g_tracer_type), pointer">
  !   Pointer to the head of generic tracer list.
  !  </IN>
  !  <IN NAME="dt" TYPE="real">
  !   Time step increment 
  !  </IN>
  !  <IN NAME="tau" TYPE="integer">
  !   Time step index to be used for %field
  !  </IN>
  ! </SUBROUTINE>
  subroutine generic_ERGOM_update_from_bottom(tracer_list, dt, tau, model_time)
    type(g_tracer_type), pointer :: tracer_list
    real,               intent(in) :: dt
    integer,            intent(in) :: tau
    type(time_type),    intent(in) :: model_time
    integer :: isc,iec, jsc,jec,isd,ied,jsd,jed,nk,ntau
    logical :: used
    real, dimension(:,:,:),pointer :: grid_tmask
    real, dimension(:,:,:,:),ALLOCATABLE :: temp_field

   end subroutine generic_ERGOM_update_from_bottom

   real function theta(x)
      real, intent(in) :: x
      if (x > 0) then
         theta=1.0
      else
         theta=0.0
      endif
      return
   end function theta

   real function power(x,y)
      real, intent(in) :: x, y
      power = x**y
      return
   end function power


  ! <SUBROUTINE NAME="generic_ERGOM_update_from_source">
  !  <OVERVIEW>
  !   Update tracer concentration fields due to the source/sink contributions.
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Currently an empty stub for CFCs.
  !  </DESCRIPTION>
  ! </SUBROUTINE>
  subroutine generic_ERGOM_update_from_source(tracer_list,Temp,Salt,rho_dzt,dzt,xt,yt,hblt_depth,&
       ilb,jlb,tau,dt,grid_dat,model_time,nbands,max_wavelength_band,sw_pen_band,opacity_band, &
       current_wave_stress)

    type(g_tracer_type),            pointer    :: tracer_list
    real, dimension(ilb:,jlb:,:),   intent(in) :: Temp,Salt,rho_dzt,dzt
    real, dimension(ilb:,jlb:),     intent(in) :: xt,yt
    real, dimension(ilb:,jlb:),     intent(in) :: hblt_depth
    integer,                        intent(in) :: ilb,jlb,tau
    real,                           intent(in) :: dt
    real, dimension(ilb:,jlb:),     intent(in) :: grid_dat
    type(time_type),                intent(in) :: model_time

    integer,                        intent(in) :: nbands
    real, dimension(:),             intent(in) :: max_wavelength_band
    real, dimension(:,ilb:,jlb:),   intent(in) :: sw_pen_band
    real, dimension(:,ilb:,jlb:,:), intent(in) :: opacity_band
    real, dimension(ilb:,jlb:),     intent(in) :: current_wave_stress

    ! abiotic parameters for biological processes
    real :: cgt_temp                ! potential temperature     [Celsius]
    real :: cgt_sali                ! salinity                  [g/kg]
    real :: cgt_light               ! downward light flux (PAR) [W/m2]
    real :: cgt_cellheight          ! cell height               [m]
    real :: cgt_density             ! density                   [kg/m3]
    real :: cgt_bottomdepth         ! bottom depth              [m]
    real :: cgt_timestep            ! timestep in days          [days]
    real :: cgt_current_wave_stress ! combined bottom stress of waves and current [N/m2]
    real :: cgt_longitude           ! longitude of grid cell    [deg]
    real :: cgt_latitude            ! latitude of grid cell     [deg]
    integer :: cgt_iteration        ! current number of iteration in the iterative loop
    integer :: cgt_year, cgt_dayofyear, cgt_mymonth, cgt_myday, cgt_myhour, cgt_myminute, cgt_mysecond ! some date/time variables
    real :: cgt_hour                ! fractional hour (0..23.9999)
            
    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_update_from_source'
    integer :: isc,iec, jsc,jec,isd,ied,jsd,jed,nk,ntau, i, j, k , kblt, n, m, nband
    real    :: light_itemp, light_ntemp
    real, dimension(:,:,:) ,pointer :: grid_tmask
    integer, dimension(:,:),pointer :: mask_coast,grid_kmt
    logical                         :: used
    integer                         :: number_of_loop

    real                            :: temp1
    real                            :: temp2
    real                            :: temp3
    real                            :: temp4
    real                            :: temp5
    real                            :: temp6
    real                            :: temp7
    real                            :: temp8
    real                            :: temp9

<auxiliaries>
    real                            :: <name>   ! <description>
</auxiliaries>

<tracers>
    real                            :: <name>   ! <description>
    real                            :: above_<name>   ! value one grid cell above
  <limitations>
    real                            :: <name> 
  </limitations>
</tracers> 

<processes>
    real                            :: <name>   ! <description>
</processes>

<tracers vertLoc=WAT; riverDep/=0>
    real, dimension(:,:)   ,pointer :: runoff_flux_<name>
    real, dimension(:,:)   ,pointer :: diffusive_flux_<name>
</tracers>
<tracers vertLoc=WAT; atmosDep/=0>
    real, dimension(:,:)   ,pointer :: wet_<name>
</tracers>

    ! the following variables are used for the positive-definite scheme
    real :: timestep_fraction           ! ratio between allowed timestep and attempted timestep
    real :: timestep_fraction_new       ! ratio between timestep allowed by each tracer and attempted timestep
    real :: fraction_of_total_timestep  ! ratio between remaining timestep and original timestep
<tracers>
    real :: change_of_<name>            ! possible change during remaining timestep
</tracers>
<processes>
    real :: total_rate_<name>           ! total process rate accumulated during all sub-timesteps
</processes>
    integer :: which_tracer_exhausted ! stores the name of the exhausted tracer  
    
    call mpp_clock_begin(id_source)
    call mpp_clock_begin(id_preloop)

    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau,&
         grid_tmask=grid_tmask,grid_mask_coast=mask_coast,grid_kmt=grid_kmt)
    
    !Get all 2d tracer values
    call user_get_2d_tracer_values(tracer_list,isd,ied,jsd,jed,nk) 

    !Get the pointers to the prognostic tracers after applying the physics.
<tracers vertLoc=WAT; vertSpeed=0>
    call g_tracer_get_pointer(tracer_list,'<name>'  ,'field',ergom%<name>    )
</tracers>
<auxiliaries vertLoc=WAT; isUsedElsewhere=1>
    call g_tracer_get_values(tracer_list,'<name>'  ,'field',ergom%<name>,isd,jsd,ntau=1)
</auxiliaries>
    do n=1, NUM_SPM
      call g_tracer_get_pointer(tracer_list,trim(array_spm(n)%name)   ,'field',array_spm(n)%p_wat     )
    enddo

    ! get the pointers to vertical velocity and diffusivity fields
<tracers vertLoc=WAT; vertSpeed/=0>
    call g_tracer_get_pointer(tracer_list,'<name>','vmove',ergom%vmove_<name>)
    call g_tracer_get_pointer(tracer_list,'<name>','vdiff',ergom%vdiff_<name>)
</tracers>

    ! calculate total element concentrations in the water column
    do k = 1, nk
       do j = jsc, jec
          do i = isc, iec
          <celements>
             ergom%<total>(i,j,k,tau) = &
             <containingTracers vertLoc=WAT; vertSpeed=0>
                max(0.0,ergom%<ct>(i,j,k,tau))*<ctAmount> + &
             </containingTracers>
             <containingTracers vertLoc=WAT; vertSpeed/=0>
                array_spm(<ctNumMoving>)%p_wat(i,j,k,tau)*<ctAmount> + &
             </containingTracers>
                0.0
          </celements>
          enddo
       enddo
    enddo

    ! calculate total colored element concentrations at bottom
    do j = jsc, jec
       do i = isc, iec
       <celements>
          array_sed(<totalBottomNumFlat>)%f_sed(i,j,1) = &
          <containingTracers vertLoc=SED>
             max(0.0,array_sed(<ctNumFlat>)%f_sed(i,j,1))*<ctAmount> + &
          </containingTracers>
             0.0
       </celements>
       enddo
    enddo    

! light calculation

    nband = 1
    if (.NOT. genus_style_par) then
       k = 1
       do j = jsc, jec ; do i = isc, iec  !{
         ergom%irr_inst(i,j,k) = 0.5 * sw_pen_band(nband,i,j) * exp(-opacity_band(nband,i,j,k)* dzt(i,j,k)*0.5)       
       enddo; enddo  !} i,j,k
       do k = 2, nk ; do j = jsc, jec ; do i = isc, iec  !{
         ergom%irr_inst(i,j,k) = ergom%irr_inst(i,j,k-1) * exp(-opacity_band(nband,i,j,k)* dzt(i,j,k))       
       enddo; enddo ; enddo  !} i,j,k
    else
       k = 1
       do j = jsc, jec ; do i = isc, iec  !{
         light_itemp = opacity_band(nband,i,j,k)* dzt(i,j,k)
         light_ntemp = exp(-light_itemp)
         ergom%irr_zw(i,j,k)   = 0.5 * sw_pen_band(nband,i,j) * light_ntemp       
         ergom%irr_inst(i,j,k) = 0.5 * sw_pen_band(nband,i,j) * ( 1. -  light_ntemp ) / light_itemp      
       enddo; enddo  !} i,j
       do k = 2, nk ; do j = jsc, jec ; do i = isc, iec  !{
         light_itemp = opacity_band(nband,i,j,k)* dzt(i,j,k) + 1.0e-30  ! some schemes give zero opa. in the nk-layer
         light_ntemp = exp(-light_itemp)
         ergom%irr_zw(i,j,k)   = ergom%irr_zw(i,j,k-1) * light_ntemp       
         ergom%irr_inst(i,j,k) = ergom%irr_zw(i,j,k-1) * ( 1.-  light_ntemp ) / light_itemp      
       enddo; enddo ; enddo  !} i,j,k
    endif

    !------------------------------------
    ! STEP 1: calculate space-independent forcing variables
    !------------------------------------
     
    call get_date(model_time,cgt_year,cgt_mymonth,cgt_myday,cgt_myhour,cgt_myminute,cgt_mysecond)
    cgt_hour = cgt_myhour + cgt_myminute/60 + cgt_mysecond/3600
    cgt_myhour=days_in_year(model_time)
    if (cgt_myhour .eq. 366) then
       if (cgt_mymonth .eq.  1) cgt_dayofyear = cgt_myday
       if (cgt_mymonth .eq.  2) cgt_dayofyear = cgt_myday +  31
       if (cgt_mymonth .eq.  3) cgt_dayofyear = cgt_myday +  60
       if (cgt_mymonth .eq.  4) cgt_dayofyear = cgt_myday +  91
       if (cgt_mymonth .eq.  5) cgt_dayofyear = cgt_myday + 121
       if (cgt_mymonth .eq.  6) cgt_dayofyear = cgt_myday + 152
       if (cgt_mymonth .eq.  7) cgt_dayofyear = cgt_myday + 182
       if (cgt_mymonth .eq.  8) cgt_dayofyear = cgt_myday + 213
       if (cgt_mymonth .eq.  9) cgt_dayofyear = cgt_myday + 244
       if (cgt_mymonth .eq. 10) cgt_dayofyear = cgt_myday + 274
       if (cgt_mymonth .eq. 11) cgt_dayofyear = cgt_myday + 305
       if (cgt_mymonth .eq. 12) cgt_dayofyear = cgt_myday + 335      
    else
       if (cgt_mymonth .eq.  1) cgt_dayofyear = cgt_myday
       if (cgt_mymonth .eq.  2) cgt_dayofyear = cgt_myday +  31
       if (cgt_mymonth .eq.  3) cgt_dayofyear = cgt_myday +  59
       if (cgt_mymonth .eq.  4) cgt_dayofyear = cgt_myday +  90
       if (cgt_mymonth .eq.  5) cgt_dayofyear = cgt_myday + 120
       if (cgt_mymonth .eq.  6) cgt_dayofyear = cgt_myday + 151
       if (cgt_mymonth .eq.  7) cgt_dayofyear = cgt_myday + 181
       if (cgt_mymonth .eq.  8) cgt_dayofyear = cgt_myday + 212
       if (cgt_mymonth .eq.  9) cgt_dayofyear = cgt_myday + 243
       if (cgt_mymonth .eq. 10) cgt_dayofyear = cgt_myday + 273
       if (cgt_mymonth .eq. 11) cgt_dayofyear = cgt_myday + 304
       if (cgt_mymonth .eq. 12) cgt_dayofyear = cgt_myday + 334 
    endif
    
    !------------------------------------
    ! STEP 2: initialize isZIntegral=1 auxiliaries with zero
    !------------------------------------
    <auxiliaries isZIntegral=1; calcAfterProcesses=0>
       ergom%zintegralarray_<name> = 0.0 
    </auxiliaries>

    !------------------------------------
    ! STEP 3: Pre-loop where the isZIntegral=1 auxiliaries are calculated
    !------------------------------------
    cgt_bottomdepth = 0.0
    do k = 1, nk
       cgt_bottomdepth = cgt_bottomdepth + dzt(isc,jsc,k)
       do j = jsc, jec
          do i = isc, iec
          
             !------------------------------------
             ! STEP 3.1: prepare abiotic parameters
             !------------------------------------
             cgt_temp       = temp(i,j,k)               ! potential temperature     [Celsius]
             cgt_sali       = salt(i,j,k)               ! salinity                  [g/kg]
             cgt_light      = ergom%irr_inst(i,j,k)     ! light intensity           [W/m2]
             cgt_cellheight = dzt(i,j,k)                ! cell height               [m]
             cgt_density    = rho_dzt(i,j,k)/dzt(i,j,k) ! density [kg/m3]
             cgt_timestep   = dt/(24*3600)              ! timestep in days
             cgt_longitude  = xt(i,j)                   ! longitude [deg]
             cgt_latitude   = yt(i,j)                   ! latitude [deg]
             if (k == grid_kmt(i,j)) then
                cgt_current_wave_stress=current_wave_stress(i,j)
             endif                     
             
             !------------------------------------
             ! STEP 3.2: load tracer values
             !------------------------------------
<tracers vertLoc=WAT; vertSpeed=0; calcBeforeZIntegral=1>
             <name> = ergom%<name>(i,j,k,tau) ! <description>
             if (k .eq. 1) then
                above_<name> = ergom%<name>(i,j,k,tau)
             else
                above_<name> = ergom%<name>(i,j,k-1,tau)
             endif
</tracers>             
<tracers vertLoc=WAT; vertSpeed/=0; calcBeforeZIntegral=1>
             <name> = array_spm(<numMoving>)%p_wat(i,j,k,tau) ! <description>
             if (k .eq. 1) then
                above_<name> = array_spm(<numMoving>)%p_wat(i,j,k,tau)
             else
                above_<name> = array_spm(<numMoving>)%p_wat(i,j,k-1,tau)
             endif
</tracers>
<tracers vertLoc=FIS; calcBeforeZIntegral=1>
             <name> = array_sed(<numFlat>)%f_sed(i,j,1) ! <description>
</tracers>
<auxiliaries vertLoc=WAT; isUsedElsewhere=1; calcBeforeZIntegral=1>
             <name> = ergom%<name>(i,j,k) ! <description>
</auxiliaries>

<tracers vertLoc=WAT; isPositive=1; calcBeforeZIntegral=1>
             <name>       = max(<name>,0.0)
             above_<name> = max(above_<name>,0.0)
</tracers>
<tracers vertLoc=FIS; isPositive=1; calcBeforeZIntegral=1>
             <name>       = max(<name>,0.0)
</tracers>

             if (k == grid_kmt(i,j)) then
<tracers vertLoc=SED>
                <name> = array_sed(<numFlat>)%f_sed(i,j,1) ! <description>
</tracers>

<tracers vertLoc=SED; isPositive=1>
                <name> = max(<name>,0.0)
</tracers>
             endif

             !------------------------------------
             ! STEP 3.3: calculate auxiliaries
             !------------------------------------
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=1; calcBeforeZIntegral=1>
             ! <description> :
             <name> = (above_<formula>-<formula>)/cgt_cellheight

</auxiliaries>

             ! initialize auxiliaries for iterative loop
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=0; calcBeforeZIntegral=1; iterations/=0>
             <name> = <iterInit>
</auxiliaries>

             ! iterative loop follows
             do cgt_iteration=1,<maxIterations>
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=0; calcBeforeZIntegral=1; iterations/=0>
                if (cgt_iteration .le. <iterations>) then
                   ! <description> :
                   temp1 = <temp1>
                   temp2 = <temp2>
                   temp3 = <temp3>
                   temp4 = <temp4>
                   temp5 = <temp5>
                   temp6 = <temp6>
                   temp7 = <temp7>
                   temp8 = <temp8>
                   temp9 = <temp9>
                   <name> = <formula>
                endif
</auxiliaries>
             enddo

             ! normal auxiliaries (not iterative)
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=0; calcBeforeZIntegral=1; iterations=0>
             ! <description> :
             temp1 = <temp1>
             temp2 = <temp2>
             temp3 = <temp3>
             temp4 = <temp4>
             temp5 = <temp5>
             temp6 = <temp6>
             temp7 = <temp7>
             temp8 = <temp8>
             temp9 = <temp9>
             <name> = <formula>

</auxiliaries>

             if (k == grid_kmt(i,j)) then
                ! initialize auxiliaries for iterative loop
<auxiliaries vertLoc=SED; calcAfterProcesses=0; isZIntegral=0; calcBeforeZIntegral=1; iterations/=0>
                <name> = <iterInit>
</auxiliaries>

                ! iterative loop follows
                do cgt_iteration=1,<maxIterations>
<auxiliaries vertLoc=SED; calcAfterProcesses=0; isZIntegral=0; calcBeforeZIntegral=1; iterations/=0>
                   if (cgt_iteration .le. <iterations>) then
                      ! <description> :
                      temp1 = <temp1>
                      temp2 = <temp2>
                      temp3 = <temp3>
                      temp4 = <temp4>
                      temp5 = <temp5>
                      temp6 = <temp6>
                      temp7 = <temp7>
                      temp8 = <temp8>
                      temp9 = <temp9>
                      <name> = <formula>
                   endif
</auxiliaries>
                enddo

                ! normal auxiliaries (not iterative)
<auxiliaries vertLoc=SED; calcAfterProcesses=0; isZIntegral=0; calcBeforeZIntegral=1; iterations=0>
                ! <description> :
                temp1 = <temp1>
                temp2 = <temp2>
                temp3 = <temp3>
                temp4 = <temp4>
                temp5 = <temp5>
                temp6 = <temp6>
                temp7 = <temp7>
                temp8 = <temp8>
                temp9 = <temp9>
                <name> = <formula> 
                
</auxiliaries>
             endif
             
             if (k == 1) then
                ! initialize auxiliaries for iterative loop
<auxiliaries vertLoc=SUR; calcAfterProcesses=0; calcBeforeZIntegral=1; iterations/=0>
                <name> = <iterInit>
</auxiliaries>

                ! iterative loop follows
                do cgt_iteration=1,<maxIterations>
<auxiliaries vertLoc=SUR; calcAfterProcesses=0; calcBeforeZIntegral=1; iterations/=0>
                   if (cgt_iteration .le. <iterations>) then
                      ! <description> :
                      temp1 = <temp1>
                      temp2 = <temp2>
                      temp3 = <temp3>
                      temp4 = <temp4>
                      temp5 = <temp5>
                      temp6 = <temp6>
                      temp7 = <temp7>
                      temp8 = <temp8>
                      temp9 = <temp9>
                      <name> = <formula>
                   endif
</auxiliaries>
                enddo

                ! normal auxiliaries (not iterative)
<auxiliaries vertLoc=SUR; calcAfterProcesses=0; calcBeforeZIntegral=1; iterations=0>
                ! <description> :
                temp1 = <temp1>
                temp2 = <temp2>
                temp3 = <temp3>
                temp4 = <temp4>
                temp5 = <temp5>
                temp6 = <temp6>
                temp7 = <temp7>
                temp8 = <temp8>
                temp9 = <temp9>
                <name> = <formula> 
                
</auxiliaries>
             endif

             !------------------------------------
             ! STEP 3.4: add contribution of the current layer k to the zIntegral
             !------------------------------------
<auxiliaries isZIntegral=1; calcAfterProcesses=0>             
             ergom%zintegralarray_<name>(i,j) = ergom%zintegralarray_<name>(i,j) + (<formula>)*cgt_cellheight*cgt_density
</auxiliaries>

          enddo
       enddo
    enddo

    call mpp_clock_end(id_preloop)
    call mpp_clock_begin(id_mainloop)

    !------------------------------------
    ! STEP 4: initializations before the main loop
    !------------------------------------
    
    !------------------------------------
    ! STEP 4.1: initialize isZIntegral=1 auxiliaries which are calculated after the process rates with zero
    !------------------------------------

<auxiliaries isZIntegral=1; calcAfterProcesses=1>
       ergom%zintegralarray_<name> = 0.0 
</auxiliaries>

    !------------------------------------
    ! STEP 4.2: initialize the arrays which cumulate the change of vertLoc=FIS tracers with zero
    !------------------------------------
<tracers vertLoc=FIS>
    ergom%cumulated_change_of_<name> = 0.0
</tracers>    

    !------------------------------------
    ! STEP 5: Main loop
    !------------------------------------

    cgt_bottomdepth = 0.0
    do k = 1, nk
       cgt_bottomdepth = cgt_bottomdepth + dzt(isc,jsc,k)
       do j = jsc, jec
          do i = isc, iec
          
             !------------------------------------
             ! STEP 5.1: prepare abiotic parameters
             !------------------------------------
             cgt_temp       = temp(i,j,k)               ! potential temperature     [Celsius]
             cgt_sali       = salt(i,j,k)               ! salinity                  [g/kg]
             cgt_light      = ergom%irr_inst(i,j,k)     ! light intensity           [W/m2]
             cgt_cellheight = dzt(i,j,k)                ! cell height               [m]
             cgt_density    = rho_dzt(i,j,k)/dzt(i,j,k) ! density [kg/m3]
             cgt_timestep   = dt/(24*3600)              ! timestep in days
             cgt_longitude  = xt(i,j)                   ! longitude [deg]
             cgt_latitude   = yt(i,j)                   ! latitude [deg]
             if (k == grid_kmt(i,j)) then
                cgt_current_wave_stress=current_wave_stress(i,j)
             endif                     
             
             !------------------------------------
             ! STEP 5.2: load tracer values
             !------------------------------------
<tracers vertLoc=WAT; vertSpeed=0>
             <name> = ergom%<name>(i,j,k,tau) ! <description>
             if (k .eq. 1) then
                above_<name> = ergom%<name>(i,j,k,tau)
             else
                above_<name> = ergom%<name>(i,j,k-1,tau)
             endif
</tracers>             
<tracers vertLoc=WAT; vertSpeed/=0>
             <name> = array_spm(<numMoving>)%p_wat(i,j,k,tau) ! <description>
             if (k .eq. 1) then
                above_<name> = array_spm(<numMoving>)%p_wat(i,j,k,tau)
             else
                above_<name> = array_spm(<numMoving>)%p_wat(i,j,k-1,tau)
             endif
</tracers>
<tracers vertLoc=FIS>
             <name> = array_sed(<numFlat>)%f_sed(i,j,1) ! <description>
</tracers>
<auxiliaries vertLoc=WAT; isUsedElsewhere=1>
             <name> = ergom%<name>(i,j,k) ! <description>
</auxiliaries>

<tracers vertLoc=WAT; isPositive=1>
             <name>       = max(<name>,0.0)
             above_<name> = max(above_<name>,0.0)
</tracers>
<tracers vertLoc=FIS; isPositive=1>
             <name>       = max(<name>,0.0)
</tracers>

             if (k == grid_kmt(i,j)) then
<tracers vertLoc=SED>
                <name> = array_sed(<numFlat>)%f_sed(i,j,1) ! <description>
</tracers>

<tracers vertLoc=SED; isPositive=1>
                <name> = max(<name>,0.0)
</tracers>
             endif

             !------------------------------------
             ! STEP 5.3: opacity calculation
             !------------------------------------
             ! some tracers contribute to the opacity
             ! opacity in m**2/mol, concentration in mol/kg -> result in m**2/kg
             ergom%bio_opacity(i,j,k) = (0.0 &
             <tracers vertLoc=WAT; opacity/=0>
                + <opacity> * <name> &
             </tracers>
                )

             ! now, convert it from m**2/kg to 1/m
             ergom%bio_opacity(i,j,k) = ergom%bio_opacity(i,j,k) * ergom%rho_0          

             !------------------------------------
             ! STEP 5.4.1: calculate auxiliaries
             !------------------------------------
<auxiliaries isZIntegral=1; calcAfterProcesses=0>
             ! <description> :
             <name> = ergom%zintegralarray_<name>(i,j)
</auxiliaries>
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=1>
             ! <description> :
             <name> = (above_<formula>-<formula>)/cgt_cellheight

</auxiliaries>

             ! initialize auxiliaries for iterative loop
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=0; iterations/=0>
             <name> = <iterInit>
</auxiliaries>

             ! iterative loop follows
             do cgt_iteration=1,<maxIterations>
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=0; iterations/=0>
                if (cgt_iteration .le. <iterations>) then
                   ! <description> :
                   temp1 = <temp1>
                   temp2 = <temp2>
                   temp3 = <temp3>
                   temp4 = <temp4>
                   temp5 = <temp5>
                   temp6 = <temp6>
                   temp7 = <temp7>
                   temp8 = <temp8>
                   temp9 = <temp9>
                   <name> = <formula>
                endif
</auxiliaries>
             enddo

             ! normal auxiliaries (not iterative)
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isZGradient=0; iterations=0>
             ! <description> :
             temp1 = <temp1>
             temp2 = <temp2>
             temp3 = <temp3>
             temp4 = <temp4>
             temp5 = <temp5>
             temp6 = <temp6>
             temp7 = <temp7>
             temp8 = <temp8>
             temp9 = <temp9>
             <name> = <formula>

</auxiliaries>

             if (k == grid_kmt(i,j)) then
                ! initialize auxiliaries for iterative loop
<auxiliaries vertLoc=SED; calcAfterProcesses=0; isZIntegral=0; iterations/=0>
                <name> = <iterInit>
</auxiliaries>

                ! iterative loop follows
                do cgt_iteration=1,<maxIterations>
<auxiliaries vertLoc=SED; calcAfterProcesses=0; isZIntegral=0; iterations/=0>
                   if (cgt_iteration .l. <iterations>) then
                      ! <description> :
                      temp1 = <temp1>
                      temp2 = <temp2>
                      temp3 = <temp3>
                      temp4 = <temp4>
                      temp5 = <temp5>
                      temp6 = <temp6>
                      temp7 = <temp7>
                      temp8 = <temp8>
                      temp9 = <temp9>
                      <name> = <formula>
                   endif
</auxiliaries>
                enddo

                ! normal auxiliaries (not iterative)
<auxiliaries vertLoc=SED; calcAfterProcesses=0; isZIntegral=0; iterations=0>
                ! <description> :
                temp1 = <temp1>
                temp2 = <temp2>
                temp3 = <temp3>
                temp4 = <temp4>
                temp5 = <temp5>
                temp6 = <temp6>
                temp7 = <temp7>
                temp8 = <temp8>
                temp9 = <temp9>
                <name> = <formula> 
                
</auxiliaries>
             endif
             
             if (k == 1) then
                ! initialize auxiliaries for iterative loop
<auxiliaries vertLoc=SUR; calcAfterProcesses=0; iterations/=0>
                <name> = <iterInit>
</auxiliaries>

                ! iterative loop follows
                do cgt_iteration=1,<maxIterations>
<auxiliaries vertLoc=SUR; calcAfterProcesses=0; iterations/=0>
                   if (cgt_iteration .le. <iterations>) then
                      ! <description> :
                      temp1 = <temp1>
                      temp2 = <temp2>
                      temp3 = <temp3>
                      temp4 = <temp4>
                      temp5 = <temp5>
                      temp6 = <temp6>
                      temp7 = <temp7>
                      temp8 = <temp8>
                      temp9 = <temp9>
                      <name> = <formula>
                   endif
</auxiliaries>
                enddo

                ! normal auxiliaries (not iterative)
<auxiliaries vertLoc=SUR; calcAfterProcesses=0; iterations=0>
                ! <description> :
                temp1 = <temp1>
                temp2 = <temp2>
                temp3 = <temp3>
                temp4 = <temp4>
                temp5 = <temp5>
                temp6 = <temp6>
                temp7 = <temp7>
                temp8 = <temp8>
                temp9 = <temp9>
                <name> = <formula> 
                
</auxiliaries>
             endif
             
             !------------------------------------
             ! STEP 5.4.2: output of auxiliaries
             !------------------------------------
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isUsedElsewhere=0>             
             if (ergom%id_<name> .gt. 0) then
               ergom%<name>(i,j,k) = <name>
             endif 
</auxiliaries>
<auxiliaries vertLoc=WAT; calcAfterProcesses=0; isUsedElsewhere=1>             
             ergom%<name>(i,j,k) = <name>
</auxiliaries>
             if (k == grid_kmt(i,j)) then
<auxiliaries vertLoc=SED; calcAfterProcesses=0>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = <name>
                endif
</auxiliaries>
             endif
             if (k == 1) then
<auxiliaries vertLoc=SUR; calcAfterProcesses=0>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = <name>
                endif
</auxiliaries>
             endif

             !------------------------------------
             ! STEP 5.5: calculate process limitations
             !------------------------------------
<tracers vertLoc=WAT>
  <limitations>
             <name> = <formula>
  </limitations>
</tracers>

             if (k == grid_kmt(i,j)) then
<tracers vertLoc=SED>
  <limitations>
                <name> = <formula>
  </limitations>
</tracers>
             endif

             if (k == 1) then
<tracers vertLoc=SUR>
  <limitations>
                <name> = <formula>
  </limitations>
</tracers>
             endif

             !------------------------------------
             ! STEP 5.6
             !
             !-- POSITIVE-DEFINITE SCHEME --------
             !-- means the following steps will be repeated as often as nessecary
             !------------------------------------
             fraction_of_total_timestep = 1.0   ! how much of the original timestep is remaining
<processes>
             total_rate_<name>          = 0.0
</processes>
             number_of_loop = 1

             do while (cgt_timestep .gt. 0.0)

                !------------------------------------
                ! STEP 6.1: calculate process rates
                !------------------------------------
<processes vertLoc=WAT>
                ! <description> :
                <name> = <turnover>
                <name> = max(<name>,0.0)

</processes>

                if (k == grid_kmt(i,j)) then
<processes vertLoc=SED>
                   ! <description> :
                   <name> = <turnover> 
                   <name> = max(<name>,0.0)
                
</processes>
                endif
             
                if (k == 1) then
<processes vertLoc=SUR>
                   ! <description> :
                   <name> = <turnover> 
                   <name> = max(<name>,0.0)
                
</processes>
                endif

                !------------------------------------
                ! STEP 6.2: calculate possible euler-forward change (in a full timestep)
                !------------------------------------
<tracers>
                change_of_<name> = 0.0
</tracers>

<tracers vertLoc=WAT; hasRatesFlat=0>
             
                change_of_<name> = change_of_<name> + cgt_timestep*(0.0 &
                <timeTendencies vertLoc=WAT>
                   <timeTendency> & ! <description>
                </timeTendencies>
                )
</tracers>
<tracers vertLoc=FIS; hasRatesFlat=0>
             
                change_of_<name> = change_of_<name> + cgt_timestep*(0.0 &
                <timeTendencies vertLoc=WAT>
                   <timeTendency> & ! <description>
                </timeTendencies>
                )
</tracers>

                if (k == 1) then
<tracers vertLoc=WAT; hasRatesFlat=2>

                   change_of_<name> = change_of_<name> + cgt_timestep*(0.0 &
                   <timeTendencies vertLoc=SUR>
                      <timeTendency> & ! <description>
                   </timeTendencies>
                   )
</tracers>
               endif

                if (k == grid_kmt(i,j)) then
<tracers vertLoc=WAT; hasRatesFlat=1>

                   change_of_<name> = change_of_<name> + cgt_timestep*(0.0 &
                   <timeTendencies vertLoc=SED>
                      <timeTendency> & ! <description>
                   </timeTendencies>
                   )
</tracers>
<tracers vertLoc=SED; hasRates>

                   change_of_<name> = change_of_<name> + cgt_timestep*(0.0 &
                   <timeTendencies>
                      <timeTendency> & ! <description>
                   </timeTendencies>
                   )
</tracers>                         
<tracers vertLoc=FIS; hasRatesFlat=1>

                   change_of_<name> = change_of_<name> + cgt_timestep*(0.0 &
                   <timeTendencies vertLoc=SED>
                      <timeTendency> & ! <description>
                   </timeTendencies>
                   )
</tracers>

                endif           

                !------------------------------------
                ! STEP 5.6.3: calculate maximum fraction of the timestep before some tracer gets exhausted
                !------------------------------------
                timestep_fraction = 1.0
                which_tracer_exhausted = -1

                ! find the tracer which is exhausted after the shortest period of time

                ! in the water column
<tracers vertLoc=WAT; vertSpeed=0; isPositive=1>
             
                ! check if tracer <name> was exhausted from the beginning and is still consumed
                if ((ergom%<name>(i,j,k,tau) .le. 0.0) .and. (change_of_<name> .lt. 0.0)) then
                   timestep_fraction = 0.0
                   which_tracer_exhausted = <numTracer>
                endif
                ! check if tracer <name> was present, but got exhausted
                if ((ergom%<name>(i,j,k,tau) .gt. 0.0) .and. (ergom%<name>(i,j,k,tau) + change_of_<name> .lt. 0.0)) then
                   timestep_fraction_new = ergom%<name>(i,j,k,tau) / (0.0 - change_of_<name>)
                   if (timestep_fraction_new .le. timestep_fraction) then
                      which_tracer_exhausted = <numTracer>
                      timestep_fraction = timestep_fraction_new
                   endif
                endif
</tracers>
<tracers vertLoc=WAT; vertSpeed/=0; isPositive=1>

                ! check if tracer <name> was exhausted from the beginning and is still consumed
                if ((array_spm(<numMoving>)%p_wat(i,j,k,tau) .le. 0.0) .and. (change_of_<name> .lt. 0.0)) then
                   timestep_fraction = 0.0
                   which_tracer_exhausted = <numTracer>
                endif
                ! check if tracer <name> was present, but got exhausted
                if ((array_spm(<numMoving>)%p_wat(i,j,k,tau) .gt. 0.0) .and. (array_spm(<numMoving>)%p_wat(i,j,k,tau) + change_of_<name> .lt. 0.0)) then
                   timestep_fraction_new = array_spm(<numMoving>)%p_wat(i,j,k,tau) / (0.0 - change_of_<name>)
                   if (timestep_fraction_new .le. timestep_fraction) then
                      which_tracer_exhausted = <numTracer>
                      timestep_fraction = timestep_fraction_new
                   endif
                endif
</tracers>

          
                ! in the bottom layer
                if (k == grid_kmt(i,j)) then
<tracers vertLoc=SED; isPositive=1>

                   ! check if tracer <name> was exhausted from the beginning and is still consumed
                   if ((array_sed(<numFlat>)%f_sed(i,j,1) .le. 0.0) .and. (change_of_<name> .lt. 0.0)) then
                      timestep_fraction = 0.0
                      which_tracer_exhausted = <numTracer>
                   endif
                   ! check if tracer <name> was present, but got exhausted
                   if ((array_sed(<numFlat>)%f_sed(i,j,1) .gt. 0.0) .and. (array_sed(<numFlat>)%f_sed(i,j,1) + change_of_<name> .lt. 0.0)) then
                      timestep_fraction_new = array_sed(<numFlat>)%f_sed(i,j,1) / (0.0 - change_of_<name>)
                      if (timestep_fraction_new .le. timestep_fraction) then
                         which_tracer_exhausted = <numTracer>
                         timestep_fraction = timestep_fraction_new
                      endif
                   endif
</tracers>                         
                endif           

                ! now, update the limitations: rates of the processes limited by this tracer become zero in the future

<tracers isPositive=1; vertLoc/=FIS>
                if (<numTracer> .eq. which_tracer_exhausted) then
                  <limitations>
                   <name> = 0.0
                  </limitations>
                endif
</tracers>

                !------------------------------------
                ! STEP 5.6.4: apply a Euler-forward timestep with the fraction of the time
                !------------------------------------ 
                ! in the water column
<tracers vertLoc=WAT; vertSpeed=0>
             
                ! tracer <name> (<description>):
                ergom%<name>(i,j,k,tau) = ergom%<name>(i,j,k,tau) + change_of_<name> * timestep_fraction
</tracers>
<tracers vertLoc=WAT; vertSpeed/=0>

                ! tracer <name> (<description>):
                array_spm(<numMoving>)%p_wat(i,j,k,tau) = array_spm(<numMoving>)%p_wat(i,j,k,tau) + change_of_<name> * timestep_fraction
</tracers>
<tracers vertLoc=FIS>

                ! tracer <name> (<description>)
                ergom%cumulated_change_of_<name>(i,j) = ergom%cumulated_change_of_<name>(i,j) + change_of_<name> * timestep_fraction
</tracers>                         


          
                ! in the bottom layer
                if (k == grid_kmt(i,j)) then
<tracers vertLoc=SED>

                   ! tracer <name> (<description>)
                   array_sed(<numFlat>)%f_sed(i,j,1) = array_sed(<numFlat>)%f_sed(i,j,1) + change_of_<name> * timestep_fraction
</tracers>                         
                endif           

                !------------------------------------
                ! STEP 5.6.5: output of process rates
                !------------------------------------
<processes vertLoc=WAT>             
                if (ergom%id_<name> .gt. 0) then
                  total_rate_<name> = total_rate_<name> + <name> * timestep_fraction * fraction_of_total_timestep
                endif 
</processes>
                if (k == grid_kmt(i,j)) then
<processes vertLoc=SED>
                   if (ergom%id_<name> .gt. 0) then
                      total_rate_<name> = total_rate_<name> + <name> * timestep_fraction * fraction_of_total_timestep
                   endif
</processes>
                endif
                if (k == 1) then
<processes vertLoc=SUR>
                   if (ergom%id_<name> .gt. 0) then
                      total_rate_<name> = total_rate_<name> + <name> * timestep_fraction * fraction_of_total_timestep
                   endif
</processes>
                endif
             
                !------------------------------------
                ! STEP 5.6.6: set timestep to remaining timestep only
                !------------------------------------
                cgt_timestep = cgt_timestep * (1.0 - timestep_fraction)                         ! remaining timestep
                fraction_of_total_timestep = fraction_of_total_timestep * (1.0 - timestep_fraction) ! how much of the original timestep is remaining


                if (number_of_loop .gt. 100) then
                   print*,"LOOP: ",number_of_loop
                   print*,"i=",i,", j=",j,", k=",k
                 <tracers isPositive=1>
                   if (<numTracer> .eq. which_tracer_exhausted) then
                     print*,"limiting tracer: <name>"
                     print*,"  <name> = ",<name>
                     print*,"  change_of_<name> = ",change_of_<name>
                   endif
                 </tracers>
                   print*,"tracers:"
                 <tracers>
                   print*,"  <name> = ",<name>
                 </tracers>
                   print*,"changes:"
                 <tracers>
                   print*,"  change_of_<name> = ",change_of_<name>
                 </tracers>
                   print*,"limitations:"
                 <tracers>
                  <limitations>
                   print*,"  <name> = ",<name>
                  </limitations>
                 </tracers>
                   print*,"cgt_timestep: ",cgt_timestep
                   print*,"timestep_fraction: ",timestep_fraction
                   print*,"  "
                   if (number_of_loop .gt. 200) then 
                      stop
                   endif
                endif
                number_of_loop=number_of_loop+1

             enddo
             !------------------------------------
             !-- END OF POSITIVE-DEFINITE SCHEME -
             !------------------------------------  

             !------------------------------------
             ! STEP 5.7: cumulate change of vertLoc=FIS tracers over the k levels
             !------------------------------------
             if (k == grid_kmt(i,j)) then
<tracers vertLoc=FIS>
                ! apply change of <description>:
                array_sed(<numFlat>)%f_sed(i,j,1) = array_sed(<numFlat>)%f_sed(i,j,1) + cumulated_change_of_<name>
</tracers>
             endif

             !------------------------------------
             ! STEP 5.8: output of process rates
             !------------------------------------
<processes vertLoc=WAT>             
             if (ergom%id_<name> .gt. 0) then
               ergom%<name>(i,j,k) = total_rate_<name>
             endif 
</processes>
             if (k == grid_kmt(i,j)) then
<processes vertLoc=SED>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = total_rate_<name>
                endif
</processes>
             endif
             if (k == 1) then
<processes vertLoc=SUR>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = total_rate_<name>
                endif
</processes>
             endif

             !------------------------------------
             ! STEP 5.9: calculate "late" auxiliaries
             !------------------------------------
             !------------------------------------
             ! STEP 5.9.1: calculate all but the isZIntegral tracers
             !------------------------------------

<auxiliaries vertLoc=WAT; calcAfterProcesses=1; isZGradient=0>
             ! <description> :
             temp1 = <temp1>
             temp2 = <temp2>
             temp3 = <temp3>
             temp4 = <temp4>
             temp5 = <temp5>
             temp6 = <temp6>
             temp7 = <temp7>
             temp8 = <temp8>
             temp9 = <temp9>
             <name> = <formula>
             
</auxiliaries>

             if (k == grid_kmt(i,j)) then
<auxiliaries vertLoc=SED; calcAfterProcesses=1; isZIntegral=0>
                ! <description> :
                temp1 = <temp1>
                temp2 = <temp2>
                temp3 = <temp3>
                temp4 = <temp4>
                temp5 = <temp5>
                temp6 = <temp6>
                temp7 = <temp7>
                temp8 = <temp8>
                temp9 = <temp9>
                <name> = <formula> 
                
</auxiliaries>
             endif
             
             if (k == 1) then
<auxiliaries vertLoc=SUR; calcAfterProcesses=1>
                ! <description> :
                temp1 = <temp1>
                temp2 = <temp2>
                temp3 = <temp3>
                temp4 = <temp4>
                temp5 = <temp5>
                temp6 = <temp6>
                temp7 = <temp7>
                temp8 = <temp8>
                temp9 = <temp9>
                <name> = <formula> 
                
</auxiliaries>
             endif
             
             !------------------------------------
             ! STEP 5.9.2: cumulate the isZIntegral auxiliaries over the k levels
             !------------------------------------
<auxiliaries vertLoc=SED; calcAfterProcesses=1; isZIntegral=1>
             ergom%zintegralarray_<name>(i,j) = ergom%zintegralarray_<name>(i,j) + (<formula>)*cgt_cellheight*cgt_density
</auxiliaries>

             !------------------------------------
             ! STEP 5.10: output of "late" auxiliaries
             !------------------------------------
<auxiliaries vertLoc=WAT; calcAfterProcesses=1; isUsedElsewhere=0>             
             if (ergom%id_<name> .gt. 0) then
               ergom%<name>(i,j,k) = <name>
             endif 
</auxiliaries>
<auxiliaries vertLoc=WAT; calcAfterProcesses=1; isUsedElsewhere=1>             
             ergom%<name>(i,j,k) = <name>
</auxiliaries>
             if (k == grid_kmt(i,j)) then
<auxiliaries vertLoc=SED; calcAfterProcesses=1; isZIntegral=0>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = <name>
                endif
</auxiliaries>
<auxiliaries vertLoc=SED; calcAfterProcesses=1; isZIntegral=1>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = ergom%zintegralarray_<name>(i,j)
                endif
</auxiliaries>
             endif
             if (k == 1) then
<auxiliaries vertLoc=SUR; calcAfterProcesses=1>
                if (ergom%id_<name> .gt. 0) then
                   ergom%<name>(i,j) = <name>
                endif
</auxiliaries>
             endif

             !---------------------------------------
             ! STEP 5.11: passing vertical velocity and diffusivity to the coupler
             !---------------------------------------

             ! IMPLICIT MOVEMENT, uncomment this to switch it on
             <tracers vertLoc=WAT; vertSpeed/=0>
               ergom%vmove_<name>(i,j,k) = (<vertSpeed>)/(24*3600)
               ergom%vdiff_<name>(i,j,k) = <vertDiff>
             </tracers> 

             ! EXPLICIT MOVEMENT
             ! commented out = set to zero
             <tracers vertLoc=WAT; vertSpeed/=0>
                array_spm(<numMoving>)%move(i,j,k)=0.0 !(<vertSpeed>)/(24.0*3600.0)
                array_spm(<numMoving>)%diff(i,j,k)=0.0 !(<vertDiff>)
             </tracers> 

             !---------------------------------------
             ! STEP 5.12: output of parameters for surface flux of gasses
             !---------------------------------------

             ! Solubility has to be converted from mol/kg/Pa to mol/m3/Pa 
             ! Surface concentration has to be converted from mol/kg to mol/m3
             
             if (k == 1) then 
               <tracers solubility/=0>
                ergom%<gasName>_alpha(i,j) = <solubility> * ergom%Rho_0  ! solubility of <gasName> [mol/m3/Pa]
                ergom%<gasName>_csurf(i,j) = <gasName> * ergom%Rho_0  ! surface concentration of dissolved <gasName> [mol/m3]
                ergom%<gasName>_sc_no(i,j) = <schmidtNumber>  ! Schmidt number of <gasName> [1]
               </tracers>
             endif

          enddo
       enddo
    enddo

    ! send surface concentration, solubility and Schmidt number to the coupler
    <tracers solubility/=0>
    call g_tracer_set_values(tracer_list,'<name>','alpha',ergom%<gasName>_alpha ,isd,jsd)
    call g_tracer_set_values(tracer_list,'<name>','csurf',ergom%<gasName>_csurf ,isd,jsd)
    call g_tracer_set_values(tracer_list,'<name>','sc_no',ergom%<gasName>_sc_no ,isd,jsd)
    </tracers>

    call mpp_clock_end(id_mainloop)
    call mpp_clock_begin(id_vertmig)
  
    ! calculate total element concentrations, these are needed for tracer movement
    do k = 1, nk
       do j = jsc, jec
          do i = isc, iec
          <celements>
             ergom%<total>(i,j,k,tau) = &
             <containingTracers vertLoc=WAT; vertSpeed=0>
                max(0.0,ergom%<ct>(i,j,k,tau))*<ctAmount> + &
             </containingTracers>
             <containingTracers vertLoc=WAT; vertSpeed/=0>
                array_spm(<ctNumMoving>)%p_wat(i,j,k,tau)*<ctAmount> + &
             </containingTracers>
                0.0
          </celements>
          enddo
       enddo
    enddo

!    ! debug vertical movement of tracers
!    do j = jsc, jec
!       do i = isc, iec
!          if ((i .eq. 37) .and. (j .eq. 12)) then
!           <tracers vertLoc=WAT; vertSpeed=0>
!             print*,"   "
!             print*,"<name> before movement:"
!             do k = 1, nk
!                print*,"   ",k,"   ",ergom%<name>(i,j,k,tau)
!             enddo
!           </tracers>
!           <tracers vertLoc=WAT; vertSpeed/=0>
!             print*,"   "
!             print*,"<name> before movement:"
!             do k = 1, nk
!                print*,"   ",k,"   ",array_spm(<numMoving>)%p_wat(i,j,k,tau)
!             enddo
!           </tracers>
!          endif
!       enddo
!    enddo
    ! vertical movement of tracers
    do m = 1, NUM_VMOVE_STEPS
       ! store the age concentration in an array which is not modified during the movement
       <celements isAging/=0>
         ergom%old_<aged> = ergom%<aged>(:,:,:,tau)
       </celements>
       ! first, move the age concentration of marked elements
       <tracers childOf/=none; vertLoc=WAT; vertSpeed/=0; hasCeAged>
!          call generic_ERGOM_vmove(array_spm(<numMoving>)%move, &
!                                   ergom%<ceAgedName>(:,:,:,tau), ergom%old_<ceAgedName>(:,:,:,tau), &
!                                   array_spm(<numMoving>)%p_wat(:,:,:,tau)*<ceAmount>, ergom%<ceTotalName>(:,:,:,tau), &
!                                   dzt, dt/NUM_VMOVE_STEPS, isd, ied, jsd, jed)

       </tracers>
       ! second, move the tracers (including marked tracers) themselves
       <tracers vertLoc=WAT; vertSpeed/=0>
!          call generic_ERGOM_vmove(array_spm(<numMoving>)%move, &
!                                   array_spm(<numMoving>)%p_wat(:,:,:,tau), array_spm(<numMoving>)%p_wat(:,:,:,tau), &
!                                   array_spm(<numMoving>)%p_wat(:,:,:,tau), array_spm(<numMoving>)%p_wat(:,:,:,tau), &
!                                   dzt, dt/NUM_VMOVE_STEPS, isd, ied, jsd, jed)
       </tracers>
       ! third, calculate new total marked element concentrations
       do k = 1, nk
          do j = jsc, jec
             do i = isc, iec
             <celements>
                ergom%<total>(i,j,k,tau) = &
                <containingTracers vertLoc=WAT; vertSpeed=0>
                   max(0.0,ergom%<ct>(i,j,k,tau))*<ctAmount> + &
                </containingTracers>
                <containingTracers vertLoc=WAT; vertSpeed/=0>
                   array_spm(<ctNumMoving>)%p_wat(i,j,k,tau)*<ctAmount> + &
                </containingTracers>
                   0.0
             </celements>
             enddo
          enddo
       enddo
    enddo
    ! vertical diffusion of tracers
    do m = 1, NUM_VMOVE_STEPS
       ! store the age concentration in an array which is not modified during the movement
       <celements isAging/=0>
         ergom%old_<aged> = ergom%<aged>(:,:,:,tau)
       </celements>
       ! first, diffuse the age concentration of marked elements
       <tracers childOf/=none; vertLoc=WAT; vertSpeed/=0; hasCeAged>
!          call generic_ERGOM_vdiff(array_spm(<numMoving>)%diff, &
!                                   ergom%<ceAgedName>(:,:,:,tau), ergom%old_<ceAgedName>(:,:,:,tau),&
!                                   array_spm(<numMoving>)%p_wat(:,:,:,tau)*<ceAmount>, ergom%<ceTotalName>(:,:,:,tau), &
!                                   dzt, dt/NUM_VMOVE_STEPS, isd, ied, jsd, jed)

       </tracers>
       ! second, diffuse the tracers (including marked tracers) themselves
       <tracers vertLoc=WAT; vertSpeed/=0>
!          call generic_ERGOM_vdiff(array_spm(<numMoving>)%diff, &
!                                   array_spm(<numMoving>)%p_wat(:,:,:,tau), array_spm(<numMoving>)%p_wat(:,:,:,tau), &
!                                   array_spm(<numMoving>)%p_wat(:,:,:,tau), array_spm(<numMoving>)%p_wat(:,:,:,tau), &
!                                   dzt, dt/NUM_VMOVE_STEPS, isd, ied, jsd, jed)
       </tracers>
       ! third, calculate new total marked element concentrations
       do k = 1, nk
          do j = jsc, jec
             do i = isc, iec
             <celements>
                ergom%<total>(i,j,k,tau) = &
                <containingTracers vertLoc=WAT; vertSpeed=0>
                   max(0.0,ergom%<ct>(i,j,k,tau))*<ctAmount> + &
                </containingTracers>
                <containingTracers vertLoc=WAT; vertSpeed/=0>
                   array_spm(<ctNumMoving>)%p_wat(i,j,k,tau)*<ctAmount> + &
                </containingTracers>
                   0.0
             </celements>
             enddo
          enddo
       enddo
    enddo
!    ! debug vertical movement of tracers
!    do j = jsc, jec
!       do i = isc, iec
!          if ((i .eq. 37) .and. (j .eq. 12)) then
!           <tracers vertLoc=WAT; vertSpeed=0>
!             print*,"   "
!             print*,"<name> after movement:"
!             do k = 1, nk
!                print*,"   ",k,"   ",ergom%<name>(i,j,k,tau)
!             enddo
!           </tracers>
!           <tracers vertLoc=WAT; vertSpeed/=0>
!             print*,"   "
!             print*,"<name> after movement:"
!             do k = 1, nk
!                print*,"   ",k,"   ",array_spm(<numMoving>)%p_wat(i,j,k,tau)
!             enddo
!           </tracers>
!          endif
!       enddo
!    enddo

    ! calculate total colored element concentrations at bottom
    do j = jsc, jec
       do i = isc, iec
       <celements>
          array_sed(<totalBottomNumFlat>)%f_sed(i,j,1) = &
          <containingTracers vertLoc=SED>
             max(0.0,array_sed(<ctNumFlat>)%f_sed(i,j,1))*<ctAmount> + &
          </containingTracers>
             0.0
       </celements>
       enddo
    enddo

    call mpp_clock_end(id_vertmig)
    call mpp_clock_begin(id_output)

! later, erosion and sedimentation shall take place via the sediment module

!    ergom%host_biores=0.0

    ! Do sedimentation and resuspension
!    call sedimentation_and_resuspension(NUM_SPM, array_spm, NUM_SED, array_sed, &
!    isc, iec, jsc, jec, isd, ied, jsd, jed, grid_kmt, dzt, rho_dzt, tau, dt, sed_defs, &
!    current_wave_stress, ergom_biores)

    call user_set_2d_tracer_values(tracer_list, model_time)  ! save all 2d tracers
    call g_tracer_set_values(tracer_list,'chl',     'field',ergom%bio_opacity/gamma_1, isd,jsd,ntau=1)
    call g_tracer_set_values(tracer_list,'opacity_bio','field',ergom%bio_opacity, isd,jsd,ntau=1)
<auxiliaries vertLoc=WAT; isUsedElsewhere=1>
    call g_tracer_set_values(tracer_list,'<name>','field',ergom%<name>, isd,jsd,ntau=1)
</auxiliaries>

    do n=1,NUM_SED
       call g_tracer_set_values(tracer_list,trim(array_spm(n)%name),  'btf', array_spm(n)%btf ,isd,jsd)
    enddo !n
    
!   Diagnostics

<tracers vertLoc=WAT; riverDep/=0>
    call g_tracer_get_pointer(tracer_list,'<name>','runoff_tracer_flux',runoff_flux_<name>)
    call g_tracer_get_pointer(tracer_list,'<name>','drydep',diffusive_flux_<name>)
</tracers>
<tracers vertLoc=WAT; atmosDep/=0>    
    call g_tracer_get_pointer(tracer_list,'<name>','wetdep',wet_<name>)
</tracers>

<tracers vertLoc=WAT; riverDep/=0>
    if (ergom%id_runoff_flux_<name> .gt. 0)      &
       used = send_data(ergom%id_runoff_flux_<name>, runoff_flux_<name>,     &
       model_time, rmask = grid_tmask(:,:,1), & 
       is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
</tracers>

<tracers vertLoc=WAT; atmosDep/=0>
    if (ergom%id_dep_wet_<name> .gt. 0)     &
       used = send_data(ergom%id_dep_wet_<name>, wet_<name>,                 &
       model_time, rmask = grid_tmask(:,:,1), & 
       is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
</tracers>

<auxiliaries vertLoc=WAT; isUsedElsewhere=0>
    if (ergom%id_<name> .gt. 0) &
       used = send_data(ergom%id_<name>, ergom%<name>,                 &
       model_time, rmask = grid_tmask(:,:,:), & 
       is_in=isc, js_in=jsc, ks_in=1, ie_in=iec, je_in=jec, ke_in=nk)
</auxiliaries>

<auxiliaries vertLoc/=WAT>
    if (ergom%id_<name> .gt. 0) &
       used = send_data(ergom%id_<name>, ergom%<name>,                 &
       model_time, rmask = grid_tmask(:,:,1), & 
       is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
</auxiliaries>


<processes vertLoc=WAT>
    if (ergom%id_<name> .gt. 0) &
       used = send_data(ergom%id_<name>, ergom%<name>,                 &
       model_time, rmask = grid_tmask(:,:,:), & 
       is_in=isc, js_in=jsc, ks_in=1, ie_in=iec, je_in=jec, ke_in=nk)
</processes>

<processes vertLoc/=WAT>
    if (ergom%id_<name> .gt. 0) &
       used = send_data(ergom%id_<name>, ergom%<name>,                 &
       model_time, rmask = grid_tmask(:,:,1), & 
       is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
</processes>

    if (ergom%id_irr_inst .gt. 0)           &
         used = send_data(ergom%id_irr_inst,     ergom%irr_inst,             &
         model_time, rmask = grid_tmask(:,:,:),& 
         is_in=isc, js_in=jsc, ks_in=1,ie_in=iec, je_in=jec, ke_in=nk)
         
    do n=1, NUM_SPM
      if (array_spm(n)%id_jsed .gt. 0)           &
           used = send_data(array_spm(n)%id_jsed, array_spm(n)%jsed, &
           model_time, rmask = grid_tmask(:,:,1),& 
           is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
    enddo
    
    do n=1, NUM_SED
      if (array_sed(n)%id_jgain_sed .gt. 0)           &
           used = send_data(array_sed(n)%id_jgain_sed, array_sed(n)%jgain_sed, &
           model_time, rmask = grid_tmask(:,:,1),& 
           is_in=isc, js_in=jsc ,ie_in=iec, je_in=jec)
      if (array_sed(n)%id_jloss_sed .gt. 0)           &
           used = send_data(array_sed(n)%id_jloss_sed, array_sed(n)%jloss_sed, &
           model_time, rmask = grid_tmask(:,:,1),& 
           is_in=isc, js_in=jsc ,ie_in=iec, je_in=jec)
      if (array_sed(n)%id_jres .gt. 0)           &
           used = send_data(array_sed(n)%id_jres, array_sed(n)%jres, &
           model_time, rmask = grid_tmask(:,:,1),& 
           is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
      if (array_sed(n)%id_jbiores .gt. 0)           &
           used = send_data(array_sed(n)%id_jbiores, array_sed(n)%jbiores, &
           model_time, rmask = grid_tmask(:,:,1),& 
           is_in=isc, js_in=jsc, ie_in=iec, je_in=jec)
    enddo
    
    call mpp_clock_end(id_output)
    call mpp_clock_end(id_source)

    return
  end subroutine generic_ERGOM_update_from_source

  ! <SUBROUTINE NAME="generic_ERGOM_set_boundary_values">
  !  <OVERVIEW>
  !   Calculate and set coupler values at the surface / bottom
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Subroutine is empty as these values are calculated in the main routine
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_set_boundary_values(tracer_list,SST,SSS,rho,ilb,jlb,tau)
  !  </TEMPLATE>
  !  <IN NAME="tracer_list" TYPE="type(g_tracer_type), pointer">
  !   Pointer to the head of generic tracer list.
  !  </IN>
  !  <IN NAME="ilb,jlb" TYPE="integer">
  !   Lower bounds of x and y extents of input arrays on data domain
  !  </IN>
  !  <IN NAME="SST" TYPE="real, dimension(ilb:,jlb:)">
  !   Sea Surface Temperature   
  !  </IN>
  !  <IN NAME="SSS" TYPE="real, dimension(ilb:,jlb:)">
  !   Sea Surface Salinity
  !  </IN>
  !  <IN NAME="rho" TYPE="real, dimension(ilb:,jlb:,:,:)">
  !   Ocean density
  !  </IN>
  !  <IN NAME="tau" TYPE="integer">
  !   Time step index of %field
  !  </IN>
  ! </SUBROUTINE>

  !User must provide the calculations for these boundary values.
  subroutine generic_ERGOM_set_boundary_values(tracer_list,SST,SSS,rho,ilb,jlb,taum1)
    type(g_tracer_type),          pointer    :: tracer_list
    real, dimension(ilb:,jlb:),     intent(in) :: SST, SSS 
    real, dimension(ilb:,jlb:,:,:), intent(in) :: rho
    integer,                        intent(in) :: ilb,jlb,taum1    
  end subroutine generic_ERGOM_set_boundary_values
 
  ! <SUBROUTINE NAME="generic_ERGOM_vmove">
  !  <OVERVIEW>
  !   Performs vertical movement (up or down)
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Updates particulate tracer concentrations 
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_vmove
  !  </TEMPLATE>
  ! </SUBROUTINE>

  subroutine generic_ERGOM_vmove(move, field, flux_field, numerator, denominator, dzt, dt, ilb, iub, jlb, jub)
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: move        ! vertical speed [m/s]
    real, dimension(ilb:,jlb:,:) ,intent(inout) :: field       ! tracer array to be moved
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: flux_field  ! (old) tracer array from which the fluxes result
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: numerator   ! vertical transports are multiplied by this scaling factor
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: denominator ! vertical transports are divided by this scaling factor
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: dzt         ! cell heights [m]
    real,                   intent(in)          :: dt          ! timestep [s]
    integer,                intent(in)          :: ilb, iub, jlb, jub
    
    real, dimension(:,:,:)  , pointer  :: tmask
    integer, dimension(:,:) , pointer  :: kmt
    real, dimension(ilb:iub,jlb:jub)   :: ft1, ft2
    real                               :: velocity, wpos, wneg
    integer                            :: k, i, j, kp1
    integer                            :: isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau
    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_vmove'
    
    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau,&
         grid_tmask=tmask, grid_kmt=kmt)
    ft1  = 0.0
    do k = 1, nk-1      
      kp1 = k+1
      do j = jsc, jec ; do i = isc, iec 
        velocity = 0.5*move(i,j,k)
        wpos     = velocity + abs(velocity)                    ! velocity if upward, else 0.0   [m/s]
        wneg     = velocity - abs(velocity)                    ! velocity if downward, else 0.0 [m/s]
        if (numerator(i,j,k) .eq. denominator(i,j,k)) then
           ft2(i,j) = (wneg*max(flux_field(i,j,k),0.0)+wpos*max(flux_field(i,j,kp1),0.0) ) &  
                   *tmask(i,j,k)*tmask(i,j,kp1)                ! upward transport through lower boundary of the t cell [mol*m/kg/s]
        else
           ft2(i,j) = (wneg*max(flux_field(i,j,k),0.0)*max(numerator(i,j,k),0.0)/max(denominator(i,j,k),1e-20) &
                   +wpos*max(flux_field(i,j,kp1),0.0)*max(numerator(i,j,kp1),0.0)/max(denominator(i,j,kp1),1e-20) ) &  
                   *tmask(i,j,k)*tmask(i,j,kp1)                ! upward transport through lower boundary of the t cell [mol*m/kg/s]
        endif
        field(i,j,k) = field(i,j,k) - dt*tmask(i,j,k)*(ft1(i,j)-ft2(i,j))/dzt(i,j,k)  ! change in the t-cell due to transports through lower and upper boundary [mol/kg/s]
        ft1(i,j) = ft2(i,j)
      enddo; enddo 
    enddo
    k = nk
    do j = jsc, jec ; do i = isc, iec 
      field(i,j,k) = field(i,j,k) - dt*tmask(i,j,k)*ft1(i,j)/dzt(i,j,k)
    enddo; enddo
  end subroutine generic_ERGOM_vmove

  ! <SUBROUTINE NAME="generic_ERGOM_vdiff">
  !  <OVERVIEW>
  !   Performs additional vertical diffusion of tracers
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Updates particulate tracer concentrations 
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_vdiff
  !  </TEMPLATE>
  ! </SUBROUTINE>

  subroutine generic_ERGOM_vdiff(diff, field, flux_field, numerator, denominator, dzt, dt, ilb, iub, jlb, jub)
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: diff        ! vertical diffusivity [m2/s]
    real, dimension(ilb:,jlb:,:) ,intent(inout) :: field       ! tracer array to be diffused
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: flux_field  ! (old) tracer array from which the fluxes are calculated
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: numerator   ! vertical transports are multiplied by this scaling factor
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: denominator ! vertical transports are divided by this scaling factor
    real, dimension(ilb:,jlb:,:) ,intent(in)    :: dzt         ! cell heights [m]
    real,                   intent(in)          :: dt          ! timestep [s]
    integer,                intent(in)          :: ilb, iub, jlb, jub
    
    real, dimension(:,:,:)  , pointer  :: tmask
    integer, dimension(:,:) , pointer  :: kmt
    real, dimension(ilb:iub,jlb:jub)   :: ft1, ft2
    real                               :: diffusivity
    integer                            :: k, i, j, kp1
    integer                            :: isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau
    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_vdiff'
    
    call g_tracer_get_common(isc,iec,jsc,jec,isd,ied,jsd,jed,nk,ntau,&
         grid_tmask=tmask, grid_kmt=kmt)
    ft1  = 0.0
    do k = 1, nk-1      
      kp1 = k+1
      do j = jsc, jec ; do i = isc, iec 
        diffusivity = 0.5*diff(i,j,k)+0.5*diff(i,j,kp1)
        if (numerator(i,j,k) .eq. denominator(i,j,k)) then
           ft2(i,j) = (max(flux_field(i,j,kp1),0.0)-max(flux_field(i,j,k),0.0))*diffusivity/(0.5*(dzt(i,j,k)+dzt(i,j,kp1))) &  
                   *tmask(i,j,k)*tmask(i,j,kp1)                ! upward transport through lower boundary of the t cell [mol*m/kg/s]
        else
           ft2(i,j) = (max(flux_field(i,j,kp1),0.0)*max(numerator(i,j,kp1),0.0)/max(denominator(i,j,kp1),1e-20) & 
                      -max(flux_field(i,j,k),0.0)*max(numerator(i,j,k),0.0)/max(denominator(i,j,k),1e-20)) &
                      *diffusivity/(0.5*(dzt(i,j,k)+dzt(i,j,kp1))) &
                      *tmask(i,j,k)*tmask(i,j,kp1)                ! upward transport through lower boundary of the t cell [mol*m/kg/s]
        endif
        field(i,j,k) = field(i,j,k) - dt*tmask(i,j,k)*(ft1(i,j)-ft2(i,j))/dzt(i,j,k)  ! change in the t-cell due to transports through lower and upper boundary [mol/kg/s]
        ft1(i,j) = ft2(i,j)
      enddo; enddo 
    enddo
    k = nk
    do j = jsc, jec ; do i = isc, iec 
      field(i,j,k) = field(i,j,k) - dt*tmask(i,j,k)*ft1(i,j)/dzt(i,j,k)
    enddo; enddo
  end subroutine generic_ERGOM_vdiff  

  ! <SUBROUTINE NAME="generic_ERGOM_end">
  !  <OVERVIEW>
  !   End the module.
  !  </OVERVIEW>
  !  <DESCRIPTION>
  !   Deallocate all work arrays
  !  </DESCRIPTION>
  !  <TEMPLATE>
  !   call generic_ERGOM_end
  !  </TEMPLATE>
  ! </SUBROUTINE>

  subroutine generic_ERGOM_end
    character(len=fm_string_len), parameter :: sub_name = 'generic_ERGOM_end'
    call user_deallocate_arrays
  end subroutine generic_ERGOM_end
  
  subroutine user_deallocate_arrays
  integer :: n
    deallocate(ergom%irr_inst)    
    deallocate(ergom%irr_zw)
    deallocate(ergom%bio_opacity)
    deallocate(ergom%host_biores)

  <auxiliaries vertLoc=WAT; isUsedElsewhere=1>
    deallocate(ergom%<name>)
  </auxiliaries>
  <auxiliaries vertLoc=WAT; isUsedElsewhere=0>
    if (ergom%id_<name> .gt. 0) deallocate(ergom%<name>)
  </auxiliaries>
  <auxiliaries vertLoc/=WAT>
    if (ergom%id_<name> .gt. 0) deallocate(ergom%<name>)
  </auxiliaries>
  <auxiliaries isZIntegral=1>
    deallocate(ergom%zintegralarray_<name>)
  </auxiliaries>
  <tracers vertLoc=FIS>
    deallocate(ergom%cumulated_change_of_<name>)
  </tracers>
  <celements isAging/=0>
    deallocate(ergom%old_<aged>)
  </celements>

  <processes>
    if (ergom%id_<name> .gt. 0) deallocate(ergom%<name>)
  </processes>

  <tracers solubility/=0>
    deallocate(ergom%<gasName>_alpha)
    deallocate(ergom%<gasName>_csurf)
    deallocate(ergom%<gasName>_sc_no)
  </tracers>

    do n = 1, NUM_SPM
        deallocate(array_spm(n)%move)
        deallocate(array_spm(n)%diff)
        deallocate(array_spm(n)%btf)
        deallocate(array_spm(n)%jsed)
    enddo
    do n = 1, NUM_SED
        deallocate(array_sed(n)%f_sed)
        deallocate(array_sed(n)%jgain_sed)
        deallocate(array_sed(n)%jloss_sed)
        deallocate(array_sed(n)%jres)
        deallocate(array_sed(n)%jbiores)
    end do
    deallocate(tracers_2d)
  end subroutine user_deallocate_arrays

end module generic_ERGOM
